/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import org.mule.service.http.netty.impl.streaming.StatusCallback;
import org.mule.service.http.netty.impl.streaming.StreamingEntitySender;

import java.io.InputStream;

import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;

/**
 * Utility class to create different types of {@link ChannelPromise}.
 */
public final class ChannelPromises {

  private ChannelPromises() {
    // private constructor to avoid wrong instantiations.
  }

  /**
   * @param ctx            the channel handler context.
   * @param stream         the stream being sent through the ctx.
   * @param statusCallback a {@link StatusCallback} to notify success or failures.
   * @return a promise to be used on stream sending end.
   */
  public static ChannelPromise finishStreamingPromise(ChannelHandlerContext ctx, InputStream stream,
                                                      StatusCallback statusCallback) {
    return createPromise(ctx, new FinishStreamingListener(stream, statusCallback));
  }

  /**
   * @param ctx            the channel handler context.
   * @param futureListener a listener for the promise.
   * @return a {@link ChannelPromise} with the passed listener.
   */
  public static ChannelPromise createPromise(ChannelHandlerContext ctx, ChannelFutureListener futureListener) {
    ChannelPromise channelPromise = ctx.newPromise();
    channelPromise.addListener(futureListener);
    return channelPromise;
  }

  /**
   * @param ctx            the channel handler context.
   * @param entitySender   a {@link StreamingEntitySender} being used to send an http body.
   * @param statusCallback a {@link StatusCallback} to notify success or failures.
   * @return a {@link ChannelPromise} that will call the entitySender method to send the next chunk.
   */
  public static ChannelPromise sendNextChunkPromise(ChannelHandlerContext ctx, StreamingEntitySender entitySender,
                                                    StatusCallback statusCallback) {
    ChannelPromise promise = ctx.newPromise();
    promise.addListener(new SendNextChunkListener(entitySender, statusCallback));
    return promise;
  }

  /**
   * @param ctx            the channel handler context.
   * @param statusCallback a {@link StatusCallback} to notify success or failures.
   * @param isLast         whether the part of the body that the caller is sending is the last one.
   * @return a {@link ChannelPromise} that will call the statusCallback on any error, or the success if the result is successful
   *         and the isLast parameter is true.
   */
  public static ChannelPromise promiseToCallback(ChannelHandlerContext ctx, StatusCallback statusCallback, boolean isLast) {
    ChannelFutureListener futureListener = channelFuture -> {
      if (channelFuture.isSuccess()) {
        if (isLast) {
          statusCallback.onSuccess();
        }
      } else {
        statusCallback.onFailure(channelFuture.cause());
      }
    };
    return createPromise(ctx, futureListener);
  }
}
