/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.message;

import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.http.api.domain.entity.Feedable;
import org.mule.runtime.http.api.domain.entity.FeedableHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Collection;

import io.netty.handler.codec.http.HttpHeaders;
import reactor.netty.http.client.HttpClientResponse;

/**
 * Adapter from Reactor Netty abstraction of response to the Mule one. Users should prefer accessing the headers individually from
 * this entity directly instead of using {@link #getHeaders()} in order to avoid copying the headers unnecessarily.
 */
public class ReactorNettyResponseWrapper implements HttpResponse, Feedable {

  private final HttpClientResponse response;
  private final HttpEntity entity;
  private final LazyValue<MultiMap<String, String>> headersAsMultiMap;
  private final Feedable feedableDelegate;

  public ReactorNettyResponseWrapper(HttpClientResponse response, FeedableHttpEntity entity) {
    this(response, entity, entity);
  }

  public ReactorNettyResponseWrapper(HttpClientResponse response, HttpEntity entity, Feedable feedableDelegate) {
    this.response = response;
    this.entity = entity;
    this.headersAsMultiMap = new LazyValue<>(() -> headersToMultiMap(response.responseHeaders()));
    this.feedableDelegate = feedableDelegate;
  }

  @Override
  public int getStatusCode() {
    return response.status().code();
  }

  @Override
  public String getReasonPhrase() {
    return response.status().reasonPhrase();
  }

  @Override
  public HttpEntity getEntity() {
    return entity;
  }

  @Override
  public Collection<String> getHeaderNames() {
    return response.responseHeaders().names();
  }

  @Override
  public String getHeaderValue(String headerName) {
    return response.responseHeaders().getAsString(headerName);
  }

  @Override
  public String getHeaderValueIgnoreCase(String headerName) {
    return response.responseHeaders().getAsString(headerName);
  }

  @Override
  public Collection<String> getHeaderValues(String headerName) {
    return response.responseHeaders().getAll(headerName);
  }

  @Override
  public Collection<String> getHeaderValuesIgnoreCase(String headerName) {
    return response.responseHeaders().getAll(headerName);
  }

  @Override
  public MultiMap<String, String> getHeaders() {
    return headersAsMultiMap.get();
  }

  private static MultiMap<String, String> headersToMultiMap(HttpHeaders nettyHeaders) {
    var asMultiMap = new CaseInsensitiveMultiMap(!PRESERVE_HEADER_CASE);
    nettyHeaders.iteratorAsString().forEachRemaining(header -> asMultiMap.put(header.getKey(), header.getValue()));
    return asMultiMap;
  }

  @Override
  public void feed(ByteBuffer nioBuffer) throws IOException {
    feedableDelegate.feed(nioBuffer);
  }

  @Override
  public void error(Exception exception) {
    feedableDelegate.error(exception);
  }

  @Override
  public void complete() throws IOException {
    feedableDelegate.complete();
  }

  @Override
  public void completeWithTrailers(MultiMap<String, String> trailers) throws IOException {
    feedableDelegate.completeWithTrailers(trailers);
  }
}
