/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.server;

import static org.mule.service.http.netty.impl.util.MuleToNettyUtils.adaptResponseWithoutBody;

import static io.netty.handler.codec.http.LastHttpContent.EMPTY_LAST_CONTENT;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http.DefaultHttpContent;
import io.netty.handler.codec.http.DefaultLastHttpContent;

/**
 * Implementation of {@link HttpWriter} using HTTP/1.
 */
public final class Http1Writer implements HttpWriter {

  private final ChannelHandlerContext ctx;

  public Http1Writer(ChannelHandlerContext ctx) {
    this.ctx = ctx;
  }

  @Override
  public void writeResponseHeader(HttpRequest request, HttpResponse response, ChannelPromise promise) {
    var httpProtocol = request.getProtocol();
    var nettyResponse = adaptResponseWithoutBody(response, httpProtocol);
    ctx.write(nettyResponse);
  }

  @Override
  public void writeContent(ByteBuf content, boolean isLast, ChannelPromise promise) {
    if (isLast) {
      if (content.readableBytes() > 0) {
        ctx.writeAndFlush(new DefaultLastHttpContent(content), promise);
      } else {
        ctx.writeAndFlush(EMPTY_LAST_CONTENT, promise);
      }
    } else {
      ctx.writeAndFlush(new DefaultHttpContent(content), promise);
    }
  }

  @Override
  public void writeTrailers(MultiMap<String, String> trailers, ChannelPromise promise) {
    throw new UnsupportedOperationException("Trailers are not supported for HTTP/1");
  }
}
