/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.netty.impl.util;

import static io.netty.buffer.ByteBufUtil.getBytes;
import static org.slf4j.LoggerFactory.getLogger;

import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;

import javax.net.ssl.SSLHandshakeException;

import io.netty.buffer.ByteBuf;
import io.netty.handler.codec.DecoderException;
import org.slf4j.Logger;
import reactor.netty.http.client.PrematureCloseException;

public final class NettyUtils {

  private static final Logger LOGGER = getLogger(NettyUtils.class);

  private NettyUtils() {
    // private constructor to avoid wrong instantiations
  }

  /**
   * Adapts Netty's {@link ByteBuf} abstraction to NIO {@link ByteBuf}.
   *
   * @param nettyBuf Netty's {@link ByteBuf}.
   * @return NIO abstraction of the buffer
   */
  public static ByteBuffer toNioBuffer(ByteBuf nettyBuf) {
    // TODO W-19810580: This method will avoid the copy if the bytes are in heap and if the
    // buffer contains exactly the wrapped byte[] (startIdx=0 and matches the length).
    // An alternative to improve performance could be doing an adapter from Netty to NIO
    // abstraction avoiding any type of copy even for non-heap implementations.
    byte[] bytes = getBytes(nettyBuf, nettyBuf.readerIndex(), nettyBuf.readableBytes(), false);
    return ByteBuffer.wrap(bytes);
  }

  /**
   * @param exception The exception to be analyzed.
   * @return {@code true} if the passed exception has to be wrapped as a "Remotely closed" IOException, or {@code false} otherwise
   */
  public static boolean shouldWrapAsRemotelyClosed(Throwable exception) {
    if (isConnectionReset(exception)) {
      return true;
    } else if (exception instanceof ClosedChannelException && hasSuppressedSSLException(exception)) {
      return true;
    } else if (exception instanceof DecoderException) {
      // TODO: This condition does not seem to mean that the connection was always closed remotely,
      // it just works for the moment
      LOGGER.warn("Caught exception. Interpreting it as a Remotely Closed", exception);
      return true;
    } else if (exception instanceof PrematureCloseException) {
      // Thrown by reactor-netty
      return true;
    } else {
      return false;
    }
  }

  private static boolean isConnectionReset(Throwable exception) {
    if (exception == null) {
      return false;
    }

    String message = exception.getMessage();
    if (message == null) {
      return false;
    }

    return message.contains("Connection reset");
  }

  private static boolean hasSuppressedSSLException(Throwable exception) {
    return exception.getSuppressed() != null && exception.getSuppressed().length > 0
        && exception.getSuppressed()[0] instanceof SSLHandshakeException;
  }
}
