/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.common.http2;

import static org.mule.service.http.test.netty.AllureConstants.Http2Story.HTTP_2_REACTIVE;
import static org.mule.tck.junit4.matcher.Eventually.eventually;

import static java.nio.ByteBuffer.wrap;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.concurrent.TimeUnit.MILLISECONDS;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.hasToString;
import static org.hamcrest.Matchers.is;

import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.entity.FeedableHttpEntity;
import org.mule.runtime.http.api.domain.entity.HttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import io.qameta.allure.Story;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

/**
 * Tests the API for sending and consuming entities reactively, both server-to-client and client-to-server.
 */
@Story(HTTP_2_REACTIVE)
class Http2ReactiveConsumptionTestCase extends AbstractHttp2ClientServerTestCase {

  public static final String PROGRESSIVE_RESPONSE = "/progressive-response";
  public static final String PROGRESSIVE_REQUEST = "/progressive-request";

  private TestProgressiveHttpEntityReceiver entityReceiver;
  private FeedableHttpEntity feedableEntity;

  public Http2ReactiveConsumptionTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @BeforeEach
  public void setUp() throws Exception {
    super.setUp();
    feedableEntity = service.getEntityFactory().feedable();
    entityReceiver = new TestProgressiveHttpEntityReceiver();
  }

  @Override
  protected HttpClientConfiguration.Builder configureClient(HttpClientConfiguration.Builder clientConfigBuilder) {
    return clientConfigBuilder.setStreaming(true);
  }

  @Test
  void serverRespondingProgressively() throws Exception {
    setupServerToSendEntityProgressively();

    var response = client.send(HttpRequest.builder()
        .uri(urlForPath(server, PROGRESSIVE_RESPONSE))
        .build());

    // In this test, the receiver is the client...
    entityReceiver.startReceivingEntity(response.getEntity());

    // So these sends are from server to client...
    sendString("Fragment #1\n");
    entityReceiver.assertPartialPayload("""
        Fragment #1
        """);

    sendString("Fragment #2\n");
    entityReceiver.assertPartialPayload("""
        Fragment #1
        Fragment #2
        """);

    sendString("Fragment #3\n");
    entityReceiver.assertPartialPayload("""
        Fragment #1
        Fragment #2
        Fragment #3
        """);

    feedableEntity.complete();

    assertThat(entityReceiver.getTrailers().isEmpty(), is(true));
  }

  @Test
  void clientSendingPostProgressively() throws Exception {
    setupServerToReceiveEntityReactively();

    var responseFuture = client.sendAsync(HttpRequest.builder()
        .uri(urlForPath(server, PROGRESSIVE_REQUEST))
        .method("POST")
        .entity(feedableEntity)
        .build());

    // These sends are from client to server...
    sendString("Fragment #1\n");

    // And these receptions are server-side
    entityReceiver.assertPartialPayload("""
        Fragment #1
        """);

    sendString("Fragment #2\n");
    entityReceiver.assertPartialPayload("""
        Fragment #1
        Fragment #2
        """);

    sendString("Fragment #3\n");
    entityReceiver.assertPartialPayload("""
        Fragment #1
        Fragment #2
        Fragment #3
        """);

    feedableEntity.complete();

    var response = responseFuture.get();
    assertThat(response.getStatusCode(), is(200));
  }

  /**
   * Test utility that will aggregate the contents of an {@link HttpEntity} with the reactive API
   * ({@link HttpEntity#onData(Consumer)} for data, and {@link HttpEntity#onComplete(BiConsumer)} for completion and trailers). It
   * is also a code example for an user of that API.
   */
  private static class TestProgressiveHttpEntityReceiver {

    private final StringBuilder partialPayload = new StringBuilder();
    private final CompletableFuture<MultiMap<String, String>> futureTrailers = new CompletableFuture<>();

    void startReceivingEntity(HttpEntity entity) {
      entity.onData(data -> {
        String asString = UTF_8.decode(data).toString();
        partialPayload.append(asString);
      });

      entity.onComplete((trailers, error) -> {
        if (error != null) {
          futureTrailers.completeExceptionally(error);
        } else {
          futureTrailers.complete(trailers);
        }
      });
    }

    void assertPartialPayload(String expected) {
      assertThat(partialPayload, eventually(hasToString(expected)).every(10, MILLISECONDS));
    }

    public MultiMap<String, String> getTrailers() throws ExecutionException, InterruptedException {
      return futureTrailers.get();
    }

    public void waitCompletion() {
      futureTrailers.join();
    }
  }

  private void sendString(String dataFragment) throws IOException {
    feedableEntity.feed(wrap(dataFragment.getBytes()));
  }

  private void setupServerToSendEntityProgressively() {
    server.addRequestHandler(List.of("GET"), PROGRESSIVE_RESPONSE, ((requestContext, responseCallback) -> {
      var response = HttpResponse.builder()
          .statusCode(200)
          .entity(feedableEntity)
          .build();

      responseCallback.responseReady(response, new IgnoreResponseStatusCallback());
    })).start();
  }

  private void setupServerToReceiveEntityReactively() {
    server.addRequestHandler(List.of("POST"), PROGRESSIVE_REQUEST, ((requestContext, responseCallback) -> {
      entityReceiver.startReceivingEntity(requestContext.getRequest().getEntity());
      var response = HttpResponse.builder()
          .statusCode(200)
          .build();

      entityReceiver.waitCompletion();
      responseCallback.responseReady(response, new IgnoreResponseStatusCallback());
    })).start();
  }
}
