/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 */
package org.mule.service.http.test.netty.impl.benchmark;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static io.netty.util.ReferenceCountUtil.release;
import static org.openjdk.jmh.annotations.Mode.Throughput;

import io.netty.buffer.Unpooled;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;

@BenchmarkMode(Throughput)
@OutputTimeUnit(MILLISECONDS)
@Fork(value = 1, jvmArgsAppend = {
// "-XX:+FlightRecorder",
// "-XX:StartFlightRecording=filename=cpu-profile.jfr,dumponexit=true,settings=<placeholder for your custom jfc>",
})
public class ServerPipelineBenchmark {

  @Benchmark
  @Threads(1)
  @Warmup(iterations = 1, time = 3, timeUnit = SECONDS)
  @Measurement(iterations = 2, time = 60, timeUnit = SECONDS)
  public void readRequests(ServerPipelineBenchmarkState state) {
    try {
      // The inbound pipeline will parse the raw request and call the listener, so at the end of the pipeline we won't see any
      // message.
      state.channel.writeInbound(state.getRawRequest());

      // The response message that should be written to the socket will be an outbound message, and we will see that here.
      var responseHeaderAsByteBuf = state.channel.readOutbound();

      if (responseHeaderAsByteBuf == null) {
        // Just checking for nullity here to avoid the noise in the benchmark...
        throw new IllegalStateException("No response read");
      } else {
        release(responseHeaderAsByteBuf);
      }

      // This is the payload of the response (empty for this test)
      var emptyBuf = state.channel.readOutbound();
      if (emptyBuf == null) {
        throw new IllegalStateException("Missing empty buffer!");
      } else {
        assert emptyBuf == Unpooled.EMPTY_BUFFER;
        release(emptyBuf);
      }
    } catch (Exception e) {
      throw new RuntimeException("Request execution failed", e);
    }
  }
}
