/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.functional.client;

import static java.lang.Thread.currentThread;

import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.runtime.http.api.server.HttpServer;
import org.mule.runtime.http.api.server.HttpServerConfiguration.Builder;
import org.mule.runtime.http.api.server.RequestHandler;
import org.mule.tck.junit4.rule.DynamicPort;

import org.junit.After;
import org.junit.Rule;

public abstract class AbstractHttpRedirectClientTestCase extends AbstractHttpClientTestCase {

  @Rule
  public DynamicPort redirectPort = new DynamicPort("port");

  protected HttpServer redirectServer;

  public AbstractHttpRedirectClientTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  public void setUp() throws Exception {
    redirectServer = service.getServerFactory().create(getRedirectServerConfigurationBuilder().build());
    redirectServer.start();
    RequestHandler requestHandler = getRedirectRequestHandler();

    var tccl = currentThread().getContextClassLoader();
    try {
      currentThread().setContextClassLoader(requestHandler.getContextClassLoader());
      redirectServer.addRequestHandler("/*", requestHandler);
    } finally {
      currentThread().setContextClassLoader(tccl);
    }
    currentThread().setContextClassLoader(requestHandler.getContextClassLoader());

    super.setUp();
  }

  /**
   * @return the basic configuration of the test server so subclasses can customize it
   */
  protected Builder getRedirectServerConfigurationBuilder() {
    return new Builder().setHost("localhost").setPort(redirectPort.getNumber())
        .setName("redirect-client-test-server");
  }

  /**
   * @param request the {@link HttpRequest} received by the server
   * @return the {@link HttpResponse} to return
   */
  protected abstract HttpResponse setUpHttpRedirectResponse(HttpRequest request);

  protected RequestHandler getRedirectRequestHandler() {
    return (requestContext, responseCallback) -> responseCallback
        .responseReady(setUpHttpRedirectResponse(requestContext.getRequest()),
                       new IgnoreResponseStatusCallback());
  }

  /**
   * @return the server's URI
   */
  protected String getRedirectUri() {
    return "http://localhost:" + redirectPort.getValue();
  }

  @Override
  @After
  public void tearDown() throws Exception {
    if (redirectServer != null) {
      redirectServer.stop();
      redirectServer.dispose();
    }
  }
}
