/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.benchmark;

import org.mule.runtime.http.api.Http1ProtocolConfig;
import org.mule.runtime.http.api.Http2ProtocolConfig;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.entity.EmptyHttpEntity;
import org.mule.runtime.http.api.domain.entity.InputStreamHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.tcp.TcpClientSocketProperties;

import java.io.ByteArrayInputStream;
import java.util.concurrent.CompletableFuture;

public class HttpServiceBasedClient implements TestClient {

  private final HttpClient httpClient;
  private final String serverUrl;
  private final String method;
  private final byte[] requestBody;

  public HttpServiceBasedClient(String serverUrl, String method, String protocol, byte[] requestBody, HttpService httpService) {
    this.serverUrl = serverUrl;
    this.method = method;
    this.requestBody = requestBody;

    this.httpClient = httpService.getClientFactory().create(
                                                            new HttpClientConfiguration.Builder()
                                                                .setName("benchmark-client")
                                                                .setUsePersistentConnections(true)
                                                                .setConnectionIdleTimeout(40_000)
                                                                .setMaxConnections(1024)
                                                                .setClientSocketProperties(TcpClientSocketProperties.builder()
                                                                    .keepAlive(true)
                                                                    .build())
                                                                .setHttp1Config(new Http1ProtocolConfig("HTTP/1"
                                                                    .equals(protocol)))
                                                                .setHttp2Config(new Http2ProtocolConfig("HTTP/2"
                                                                    .equals(protocol)))
                                                                .build());
  }

  @Override
  public CompletableFuture<Void> sendTestRequest() {
    var entity = requestBody.length > 0 ? new InputStreamHttpEntity(new ByteArrayInputStream(requestBody), requestBody.length)
        : new EmptyHttpEntity();

    var testRequest = HttpRequest.builder()
        .uri(serverUrl)
        .method(method)
        .entity(entity)
        .build();

    var finished = new CompletableFuture<Void>();
    httpClient.sendAsync(testRequest).whenComplete((response, throwable) -> {
      if (throwable != null) {
        finished.completeExceptionally(throwable);
      } else {
        try {
          if (response.getStatusCode() != 200) {
            throw new IllegalStateException("Response should be a 200 OK");
          }
          response.getEntity().getContent().readAllBytes();
          finished.complete(null);
        } catch (Exception e) {
          finished.completeExceptionally(e);
        }
      }
    });
    return finished;
  }


  @Override
  public void start() {
    httpClient.start();
  }

  @Override
  public void stop() {
    httpClient.stop();
  }
}
