/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.benchmark;

import static org.apache.hc.core5.http.ContentType.TEXT_PLAIN;

import java.io.IOException;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;

import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.impl.async.HttpAsyncClients;
import org.apache.hc.core5.concurrent.FutureCallback;

public class ApacheTestClient implements TestClient {

  private final CloseableHttpAsyncClient httpClient;
  private final String serverUrl;
  private final String method;
  private final byte[] requestBody;

  public ApacheTestClient(String serverUrl, String method, String protocol, byte[] requestBody) {
    this.serverUrl = serverUrl;
    this.method = method;
    this.requestBody = requestBody;
    if ("HTTP/2".equals(protocol)) {
      this.httpClient = HttpAsyncClients.createHttp2Default();
    } else {
      this.httpClient = HttpAsyncClients.createDefault();
    }
  }

  @Override
  public CompletableFuture<Void> sendTestRequest() {
    var testRequest = SimpleHttpRequest.create(method, serverUrl);
    testRequest.setBody(requestBody, TEXT_PLAIN);

    var finished = new CompletableFuture<Void>();
    httpClient.execute(testRequest, new FutureCallback<>() {

      @Override
      public void completed(SimpleHttpResponse result) {
        result.getBodyBytes();
        finished.complete(null);
      }

      @Override
      public void failed(Exception ex) {
        finished.completeExceptionally(ex);
      }

      @Override
      public void cancelled() {
        finished.completeExceptionally(new CancellationException("Cancel!!"));
      }
    });

    return finished;
  }

  @Override
  public void start() {
    httpClient.start();
  }

  @Override
  public void stop() throws IOException {
    httpClient.close();
  }
}
