/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.benchmark;

import static org.mule.runtime.api.util.DataUnit.KB;
import static org.mule.runtime.api.util.DataUnit.MB;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.service.http.impl.util.ReflectiveServiceFactory.createHttpService;

import static java.lang.Integer.getInteger;

import static org.openjdk.jmh.annotations.Level.Trial;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.http.api.HttpService;
import org.mule.tck.SimpleUnitTestSupportSchedulerService;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;

import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;

@State(Scope.Benchmark)
public class HttpClientBenchmarkExecutionPlan {

  private static final String NETTY_SERVICE_NAME = "org.mule.service.http.netty.impl.service.NettyHttpServiceImplementation";
  private static final String GRIZZLY_SERVICE_NAME = "org.mule.service.http.impl.service.HttpServiceImplementation";

  public enum ServiceConfiguration {

    NETTY_HTTP_2(NETTY_SERVICE_NAME, "HTTP/2"),

    NETTY_HTTP_1(NETTY_SERVICE_NAME, "HTTP/1"),

    GRIZZLY_HTTP_1(GRIZZLY_SERVICE_NAME, "HTTP/1"), // (no http/2 for grizzly)

    APACHE_HTTP_1(null, "HTTP/1"),

    APACHE_HTTP_2(null, "HTTP/2"),
    ;

    private final String serviceName;
    private final String protocol;

    ServiceConfiguration(String serviceName, String protocol) {
      this.serviceName = serviceName;
      this.protocol = protocol;
    }

    public String getServiceImplementation() {
      return serviceName;
    }

    public String getProtocol() {
      return protocol;
    }
  }

  public enum RequestConfiguration {

    GET("GET", 0),

    POST_1KB("POST", KB.toBytes(1)),

    POST_10KB("POST", KB.toBytes(10)),

    POST_1MB("POST", MB.toBytes(1)),

    ;

    private final String method;
    private final int bodySize;

    RequestConfiguration(String method, int bodySize) {
      this.method = method;
      this.bodySize = bodySize;
    }

    public String getMethod() {
      return method;
    }

    public int getBodySize() {
      return bodySize;
    }
  }

  private SchedulerService schedulerService;

  private HttpService httpService;

  public TestClient testClient;

  public Integer serverPort = getInteger("docker.server.port");

  public AtomicInteger totalSuccessfulRequests;

  @Param
  public ServiceConfiguration serviceConfiguration;

  @Param
  public RequestConfiguration requestConfiguration;

  @Param({"1"})
  public int requestsPerInvocation;

  @Setup(Trial)
  public void setUp() throws Exception {
    byte[] requestBody = new byte[requestConfiguration.getBodySize()];

    if (serviceConfiguration.getServiceImplementation() != null) {
      schedulerService = new SimpleUnitTestSupportSchedulerService();
      startIfNeeded(schedulerService);

      httpService = createHttpService(serviceConfiguration.getServiceImplementation(), schedulerService);
      startIfNeeded(httpService);
    }

    testClient = createTestClient(requestBody);
    testClient.start();

    totalSuccessfulRequests = new AtomicInteger(0);
  }

  @TearDown(Trial)
  public void tearDown() throws MuleException, IOException {
    if (testClient != null) {
      testClient.stop();
    }

    stopIfNeeded(httpService);
    stopIfNeeded(schedulerService);

    System.out.println("Successful requests count: " + totalSuccessfulRequests.get());
  }

  private TestClient createTestClient(byte[] requestBody) {
    String serverUrl = "http://localhost:" + serverPort;

    if (serviceConfiguration.getServiceImplementation() != null) {
      return new HttpServiceBasedClient(serverUrl, requestConfiguration.getMethod(), serviceConfiguration.getProtocol(),
                                        requestBody, httpService);
    } else {
      return new ApacheTestClient(serverUrl, requestConfiguration.getMethod(), serviceConfiguration.getProtocol(), requestBody);
    }
  }
}
