/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.service.server.grizzly;

import static java.lang.Long.parseLong;
import static org.mule.runtime.core.api.util.StringUtils.isEmpty;
import static org.mule.runtime.http.api.HttpHeaders.Names.CONTENT_LENGTH;
import static org.mule.runtime.http.api.server.HttpServerProperties.PRESERVE_HEADER_CASE;
import static org.mule.runtime.http.api.utils.HttpEncoderDecoderUtils.decodeQueryString;
import static org.mule.runtime.http.api.utils.UriCache.getUriFromString;

import org.glassfish.grizzly.http.util.MimeHeaders;
import org.mule.runtime.api.util.MultiMap;
import org.mule.runtime.http.api.domain.CaseInsensitiveMultiMap;
import org.mule.runtime.http.api.domain.HttpProtocol;
import org.mule.runtime.http.api.domain.message.BaseHttpMessage;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;

import java.net.InetSocketAddress;
import java.net.URI;
import java.util.Collection;

import org.glassfish.grizzly.http.HttpRequestPacket;
import org.glassfish.grizzly.http.Protocol;

/**
 * Base class for a grizzly based implementation of {@link BaseHttpMessage}
 *
 * @since 1.3.0
 */
public abstract class GrizzlyHttpMessage extends BaseHttpMessage implements HttpRequest {

  protected final InetSocketAddress localAddress;
  protected final boolean isTransferEncodingChunked;
  protected final long contentLength;
  protected final HttpRequestPacket requestPacket;
  private final String baseUri;

  protected String method;
  protected MultiMap<String, String> queryParams;
  protected HttpProtocol protocol;
  protected URI uri;
  protected String path;

  public GrizzlyHttpMessage(HttpRequestPacket requestPacket,
                            MultiMap<String, String> headers,
                            InetSocketAddress localAddress) {
    super(headers);
    this.requestPacket = requestPacket;
    this.localAddress = localAddress;
    isTransferEncodingChunked = requestPacket.isChunked();
    this.baseUri = getBaseProtocol() + "://" + localAddress.getHostString() + ":" + localAddress.getPort();

    long contentLengthAsLong = isTransferEncodingChunked ? -1L : 0L;
    String contentLengthAsString = requestPacket.getHeader(CONTENT_LENGTH);
    if (contentLengthAsString != null) {
      contentLengthAsLong = parseLong(contentLengthAsString);
    }
    this.contentLength = contentLengthAsLong;
  }

  @Override
  public MultiMap<String, String> getQueryParams() {
    if (queryParams == null) {
      queryParams = decodeQueryString(requestPacket.getQueryString());
    }
    return queryParams;
  }

  @Override
  public String getPath() {
    if (this.path == null) {
      this.path = requestPacket.getRequestURI();
    }
    return this.path;
  }

  @Override
  public String getMethod() {
    if (this.method == null) {
      this.method = requestPacket.getMethod().getMethodString();
    }
    return this.method;
  }

  @Override
  public HttpProtocol getProtocol() {
    if (this.protocol == null) {
      this.protocol = requestPacket.getProtocol() == Protocol.HTTP_1_0 ? HttpProtocol.HTTP_1_0 : HttpProtocol.HTTP_1_1;
    }
    return this.protocol;
  }

  @Override
  public Collection<String> getHeaderNames() {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers.keySet();
  }

  @Override
  public String getHeaderValue(String headerName) {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers.get(headerName);
  }

  @Override
  public Collection<String> getHeaderValues(String headerName) {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers.getAll(headerName);
  }

  @Override
  public MultiMap<String, String> getHeaders() {
    if (this.headers == null) {
      initializeHeaders();
    }
    return this.headers;
  }

  private void initializeHeaders() {
    this.headers = new CaseInsensitiveMultiMap(!PRESERVE_HEADER_CASE);
    MimeHeaders grizzlyHeaders = requestPacket.getHeaders();
    // For performance reasons we don't want to use Grizzly's MimeHeaders iterators because they are O(N^2)
    // The downside is that we are potentially decoding a header more than once if there are repetitions
    for (int i = 0; i < grizzlyHeaders.size(); i++) {
      String header = grizzlyHeaders.getName(i).toString();
      String value = grizzlyHeaders.getValue(i).toString();
      this.headers.put(header, value);
    }
    this.headers = this.headers.toImmutableMultiMap();
  }

  @Override
  public URI getUri() {
    if (this.uri == null) {
      String path =
          requestPacket.getRequestURI() + (isEmpty(requestPacket.getQueryString()) ? "" : "?" + requestPacket.getQueryString());
      try {
        this.uri = getUriFromString(baseUri + path);
      } catch (IllegalArgumentException e) {
        throw new IllegalArgumentException("Malformed URI: " + path);
      }
    }
    return this.uri;
  }

  protected abstract String getBaseProtocol();
}
