/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.service.client;

import static org.mule.runtime.api.util.DataUnit.KB;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import org.mule.runtime.http.api.client.HttpClient;
import org.mule.runtime.http.api.client.HttpClientConfiguration;
import org.mule.runtime.http.api.domain.entity.ByteArrayHttpEntity;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.impl.functional.client.AbstractHttpClientTestCase;
import org.junit.Test;

public class HttpClientKnownSizeTestCase extends AbstractHttpClientTestCase {

  protected static final String PAYLOAD = "Lionel Andres Messi";
  private HttpClientConfiguration.Builder clientBuilder = new HttpClientConfiguration.Builder().setName("known-size-test");

  public HttpClientKnownSizeTestCase(String serviceToLoad) {
    super(serviceToLoad);
  }

  @Override
  protected HttpResponse setUpHttpResponse(HttpRequest request) {
    return HttpResponse.builder().statusCode(OK.getStatusCode()).reasonPhrase(OK.getReasonPhrase())
        .entity(new ByteArrayHttpEntity(PAYLOAD.getBytes()))
        .build();
  }

  @Test
  public void withStreaming() throws Exception {
    testWithKnownLength(true);
  }

  @Test
  public void withoutStreaming() throws Exception {
    testWithKnownLength(false);
  }


  private void testWithKnownLength(boolean streaming) throws Exception {
    HttpClient client = service
        .getClientFactory()
        .create(clientBuilder.setResponseBufferSize(KB.toBytes(10)).setStreaming(streaming).build());
    client.start();
    try {
      HttpResponse response = client.sendAsync(getRequest(), getDefaultOptions(3000)).get();
      long sizeAsLong = PAYLOAD.length();
      assertThat(response.getEntity().getBytesLength().orElse(0), is(sizeAsLong));
    } finally {
      client.stop();
    }
  }

  protected HttpRequest getRequest() {
    return HttpRequest.builder().uri(getUri()).build();
  }

}
