/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.http.impl.service.client;

import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;

/**
 * Redirection mechanism needs to maintain certain contextual status.
 */
public class RedirectContext {

  private final List<String> seenSetCookies = new LinkedList<>();
  private int currentRedirects = 0;

  /**
   * Increments the counter of redirects until now.
   *
   * @return this same instance.
   */
  public RedirectContext incrementCurrentRedirects() {
    ++currentRedirects;
    return this;
  }

  /**
   * @return the number of times that the user called {@link #incrementCurrentRedirects()}
   */
  public int getCurrentRedirects() {
    return currentRedirects;
  }

  /**
   * In a scenario of multiple redirections, all the set-cookie values have to be considered to calculate the next redirection
   * request's cookies.
   */
  public void accumulateSeenSetCookies(Collection<String> seenSetCookies) {
    this.seenSetCookies.addAll(seenSetCookies);
  }

  /**
   * Iterates over all the cookies accumulated by {@link #accumulateSeenSetCookies}.
   *
   * @param eachCookieConsumer logic to apply to each set-cookie.
   */
  public void foreachSetCookie(Consumer<String> eachCookieConsumer) {
    this.seenSetCookies.forEach(eachCookieConsumer);
  }
}
