/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.scheduler;

import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.scheduler.SchedulerService;

/**
 * Represents the type of work that a {@link Thread} owned by a {@link Scheduler} is configured to perform.
 *
 * @since 1.0
 */
public enum ThreadType {

  /**
   * The type for {@link Thread}s managed by {@link Scheduler}s obtained with {@link SchedulerService#ioScheduler()}.
   * <p>
   * A task is considered <b>blocking I/O</b>if it spends most of it's clock duration blocked due to I/O operations.
   */
  IO("io"),

  /**
   * The type for {@link Thread}s managed by {@link Scheduler}s obtained with {@link SchedulerService#cpuIntensiveScheduler()}.
   * <p>
   * A task is considered a <b>CPU intensive</b> if its duration is more than 10 milliseconds and less than 20% of its clock time
   * is due to blocking.
   */
  CPU_INTENSIVE("cpuIntensive"),

  /**
   * The type for {@link Thread}s managed by {@link Scheduler}s obtained with {@link SchedulerService#cpuLightScheduler()}.
   * <p>
   * A task is considered <b>CPU light</b> if it doesn't block at any time and its duration is less than 10 milliseconds.
   */
  CPU_LIGHT("cpuLight"),

  /**
   * The type for {@link Thread}s managed by custom {@link Scheduler}s.
   * <p>
   * Custom threads are not managed by the Mule Runtime and not shared between different schedulers. This is for the case where
   * exclusive usage for a thread pool is needed (i.e.: NIO selectors).
   */
  CUSTOM("custom"),

  /**
   * The type for {@link Thread}s not managed by {@link Scheduler}s obtained from {@link SchedulerService}.
   */
  UNKNOWN("unknown");

  private String name;

  private ThreadType(String name) {
    this.name = name;
  }

  public String getName() {
    return name;
  }
}
