/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.service.scheduler.internal.reporting;

import static org.mule.service.scheduler.ThreadType.IO;
import static org.mule.test.allure.AllureConstants.SchedulerServiceFeature.SCHEDULER_SERVICE;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.alert.TimedDataBuffer;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import io.qameta.allure.Feature;

@Feature(SCHEDULER_SERVICE)
@ExtendWith(MockitoExtension.class)
class DefaultSchedulerViewTestCase {

  @Mock
  private ReportableScheduler scheduler;

  private DefaultSchedulerView schedulerView;

  @BeforeEach
  public void setUp() {
    schedulerView = new DefaultSchedulerView(scheduler);
  }

  @Test
  void getName() {
    when(scheduler.getName()).thenReturn("someScheduler");

    assertThat(schedulerView.getName(), is("someScheduler"));
    verify(scheduler).getName();
  }

  @Test
  void isShutdown() {
    when(scheduler.isShutdown()).thenReturn(true);

    assertThat(schedulerView.isShutdown(), is(true));
    verify(scheduler).isShutdown();
  }

  @Test
  void isTerminated() {
    when(scheduler.isTerminated()).thenReturn(true);

    assertThat(schedulerView.isTerminated(), is(true));
    verify(scheduler).isTerminated();
  }

  @Test
  void getThreadType() {
    when(scheduler.getThreadType()).thenReturn(IO);

    assertThat(schedulerView.getThreadType(), is("IO"));
    verify(scheduler).getThreadType();
  }

  @Test
  void getQueuedTasks() {
    when(scheduler.getQueuedTasks()).thenReturn(4);

    assertThat(schedulerView.getQueuedTasks(), is(4));
    verify(scheduler).getQueuedTasks();
  }

  @Test
  void getActiveTasks() {
    when(scheduler.getQueuedTasks()).thenReturn(4);
    when(scheduler.getTasksCount()).thenReturn(6);

    assertThat(schedulerView.getActiveTasks(), is(2));
    verify(scheduler).getTasksCount();
    verify(scheduler).getQueuedTasks();
  }

  @Test
  void getRejectionsOverTime() {
    var rejections = new TimedDataBuffer<String>();
    rejections.put("a");
    when(scheduler.getRejectionsOverTime()).thenReturn(rejections);

    assertThat(schedulerView.getRejectionsOverTime().forLast60MinsInterval(), is(1));
    verify(scheduler).getRejectionsOverTime();
  }

  @Test
  void getThrottlesOverTime() {
    var throttles = new TimedDataBuffer<String>();
    throttles.put("b");
    throttles.put("c");
    when(scheduler.getThrottlesOverTime()).thenReturn(throttles);

    assertThat(schedulerView.getThrottlesOverTime().forLast60MinsInterval(), is(2));
    verify(scheduler).getThrottlesOverTime();
  }
}
