package org.mule.configuration;

import java.util.HashMap;
import java.util.Map;

import org.mule.configuration.parser.ConfigurationParser;

/**
 * The {@link Configuration} class knows how to parse the two require configuration files
 * (default_sections, sections) to obtain a unique set of placeholders and sections.
 * 
 * @author damiansima
 */
public class Configuration {
	private static final String EXT_CONFIGURATION_FILE_PATH = "./doc/configReadme.xml";
	private static final String DEFAULT_CONFIGURATION_FILE_NAME = "default_sections.xml";

	private final Map<String, String> sections = new HashMap<String, String>();
	private final Map<String, String> placeholders = new HashMap<String, String>();

	private ConfigurationParser defaultConfigurationParser;
	private ConfigurationParser extensionConfigurationParser;

	public Configuration() {
		this(DEFAULT_CONFIGURATION_FILE_NAME, EXT_CONFIGURATION_FILE_PATH);
	}

	public Configuration(String defaultConfigurationFileName, String configurationFilePath) {
		try {
			defaultConfigurationParser = new ConfigurationParser(defaultConfigurationFileName);
			extensionConfigurationParser = new ConfigurationParser(configurationFilePath);
		} catch (Exception e) {
			throw new RuntimeException("There has been an error parsing the configuration file: " + e.getMessage());
		}

		initialise();
	}

	public Map<String, String> getPlaceholders() {
		return placeholders;
	}

	public Map<String, String> getSections() {
		return sections;
	}

	private void initialise() {
		populateSections();
		populatePlaceHolders();
	}

	/**
	 * It grabs all the place holders from the DEFAULT and the EXTENSION configuration and put all of
	 * them in the placeholders map.
	 * It overwrites place holder in the DEFAULT configuration with the ones in the EXTENSION
	 * configuration.
	 */
	private void populatePlaceHolders() {
		for (String key : defaultConfigurationParser.getPlaceHolders().keySet()) {
			placeholders.put(key, defaultConfigurationParser.getPlaceHolders().get(key));
		}

		for (String key : extensionConfigurationParser.getPlaceHolders().keySet()) {
			placeholders.put(key, extensionConfigurationParser.getPlaceHolders().get(key));
		}
	}

	/**
	 * It grabs all the sections from the DEFAULT and the EXTENSION configuration and put all of
	 * them in the sections map.
	 * It overwrites sections in the DEFAULT configuration with the ones in the EXTENSION
	 * configuration.
	 */
	private void populateSections() {
		for (String key : defaultConfigurationParser.getSections().keySet()) {
			sections.put(key, defaultConfigurationParser.getSections().get(key));
		}

		for (String key : extensionConfigurationParser.getSections().keySet()) {
			sections.put(key, extensionConfigurationParser.getSections().get(key));
		}
	}

}
