package org.mule.configuration.parser;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.google.common.base.Preconditions;

public class ConfigurationParser {
	private static final String SECTIONS_XPATH = "//configuration/sections/section";
	private static final String PLACEHOLDERS_XPATH = "//configuration/placeholders/placeholder";

	private XPath configurationXpath;
	private Document configurationDoc;

	private final Map<String, String> sections = new HashMap<String, String>();
	private final Map<String, String> placeholders = new HashMap<String, String>();

	public ConfigurationParser(String configurationFilePath) throws Exception {
		Preconditions.checkArgument(StringUtils.isNotBlank(configurationFilePath), "The file path can not be null nor empty.");

		configurationDoc = buildDocument(loadFile(configurationFilePath));
		configurationXpath = XPathFactory.newInstance().newXPath();

		this.parse();
	}

	public Map<String, String> getPlaceHolders() {
		return placeholders;
	}

	public String getPlaceholder(String id) {
		String value = placeholders.get(id);
		if (value != null) {
			return value;
		}
		throw new RuntimeException("The placeholder id: " + id + " is not defined in the configuration file");
	}

	public Map<String, String> getSections() {
		return sections;
	}

	public String getSection(String id) {
		String value = sections.get(id);
		if (value != null) {
			return value;
		}
		throw new RuntimeException("The section id: " + id + " is not defined in the configuration file");
	}

	private Document buildDocument(String xml) throws ParserConfigurationException, SAXException, IOException {

		DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true); // never forget this!
		DocumentBuilder builder = factory.newDocumentBuilder();
		Document doc = builder.parse(new ByteArrayInputStream(xml.getBytes()));

		return doc;
	}

	private void parse() throws Exception {
		parsePlaceholders(configurationDoc, configurationXpath);
		parseSections(configurationDoc, configurationXpath);
	}

	private void parsePlaceholders(Document doc, XPath xpath) throws Exception {

		NodeList nodeList = (NodeList) xpath.compile(PLACEHOLDERS_XPATH).evaluate(doc, XPathConstants.NODESET);
		putInMap(placeholders, nodeList);
	}

	private void parseSections(Document doc, XPath xpath) throws Exception {
		NodeList nodeList = (NodeList) xpath.compile(SECTIONS_XPATH).evaluate(doc, XPathConstants.NODESET);
		putInMap(sections, nodeList);
	}

	private void putInMap(Map<String, String> destinationMap, NodeList nodeList) throws Exception {
		int index = 0;
		while (index < nodeList.getLength()) {
			Node node = nodeList.item(index);

			String key = node.getAttributes().getNamedItem("id").getTextContent();
			String value = node.getTextContent();
			value = beautifyValue(value);
			destinationMap.put(key, value);

			index++;
		}
	}

	private String beautifyValue(String value) {
		String niceValue = value;
		String carryRet = System.getProperty("line.separator");
		niceValue = niceValue.trim();
		niceValue = StringUtils.removeStart(niceValue, carryRet);
		niceValue = StringUtils.removeEnd(niceValue, carryRet);
		return niceValue;
	}

	@SuppressWarnings("resource")
	private String loadFile(String filePath) throws IOException {
		InputStream in;

		try {
			in = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			in = Thread.currentThread().getContextClassLoader().getResourceAsStream(filePath);

			if (in == null) {
				throw new FileNotFoundException("The file " + filePath + " could not be loaded.");
			}
		}
		return IOUtils.toString(in);
	}

}
