package org.mule.core;

import org.apache.commons.lang.StringUtils;
import org.mule.configuration.Section;
import org.mule.configuration.pom.TemplatePomReader;

/**
 * The SectionSelector class contains logic on how to choose which sections of the configuration should be replaced in the template
 * Created by damiansima on 6/24/14.
 */
public class SectionSelector {

    private static final String DB_SYSTEM_ID = "db";
    private static final String SAP_SYSTEM_ID = "sap";
    private static final String SFDC_SYSTEM_ID = "sfdc";
    private static final String SIEB_SYSTEM_ID = "sieb";
    private static final String SNOW_SYSTEM_ID = "snow";
    private static final String WDAY_SYSTEM_ID = "wday";
    private static final String JOBV_SYSTEM_ID = "jobv";
    private static final String NETS_SYSTEM_ID = "nets";
    private static final String MSDYN_SYSTEM_ID = "msdyn";
    private static final String MSSP_SYSTEM_ID = "mssp";
    private static final String WDAYF_SYSTEM_ID = "wdayf";
    private static final String MKTO_SYSTEM_ID = "mkto";


    private TemplatePomReader pomReader;

    public SectionSelector() {
        this(new TemplatePomReader());
    }

    public SectionSelector(TemplatePomReader pomReader) {
        this.pomReader = pomReader;
    }

    /**
     * Validates if a section with the sectionId should be used.
     *
     * @param sectionId a string representing a sectionId
     * @return false if the section shouldn't be added to the readme
     */
    public boolean addSection(String sectionId) {

        Section section = null;
        try {
            section = Section.valueOf(sectionId.toUpperCase());
        } catch (IllegalArgumentException e) {
            return false;
        }

        if (section.evaluateSystems()) {
            String systemId = selectSystemToEvaluate(sectionId);

            if (StringUtils.isNotBlank(systemId)) {
                if (addSystemDependantSection(sectionId, systemId)) {
                    return true;
                }
            }
            return false;
        }

        return true;
    }

    private boolean addSystemDependantSection(String sectionId, String systemId) {

        if (sectionId.contains(systemId + "_")) {
            if (isSystemPresentInTemplate(systemId)) {

                if (sectionId.contains("_preconditions")) {
                    return true;
                }

                if(TemplatePomReader.BIDIRECTIONAL_PATTERN.equals(pomReader.getPattern())){
                    return true;
                }

                if (sectionId.contains("_src") && pomReader.getSourceSystem().contains(systemId)) {
                    return true;
                }

                if (sectionId.contains("_dst") && pomReader.getDestinationSystem().contains(systemId)) {
                    return true;
                }
            }
        }
        return false;
    }

    private String selectSystemToEvaluate(String sectionId) {

        if (sectionId.contains(DB_SYSTEM_ID + "_")) {
            return DB_SYSTEM_ID;
        }

        if (sectionId.contains(SAP_SYSTEM_ID + "_")) {
            return SAP_SYSTEM_ID;
        }

        if (sectionId.contains(SFDC_SYSTEM_ID + "_")) {
            return SFDC_SYSTEM_ID;
        }

        if (sectionId.contains(SIEB_SYSTEM_ID + "_")) {
            return SIEB_SYSTEM_ID;
        }

        if (sectionId.contains(SNOW_SYSTEM_ID + "_")) {
            return SNOW_SYSTEM_ID;
        }

        if (sectionId.contains(WDAY_SYSTEM_ID + "_")) {
            return WDAY_SYSTEM_ID;
        }

        if (sectionId.contains(JOBV_SYSTEM_ID + "_")) {
            return JOBV_SYSTEM_ID;
        }

        if (sectionId.contains(NETS_SYSTEM_ID + "_")) {
            return NETS_SYSTEM_ID;
        }

        if (sectionId.contains(MSDYN_SYSTEM_ID + "_")) {
            return MSDYN_SYSTEM_ID;
        }

        if (sectionId.contains(MSSP_SYSTEM_ID + "_")) {
            return MSSP_SYSTEM_ID;
        }

        if (sectionId.contains(WDAYF_SYSTEM_ID + "_")) {
            return WDAYF_SYSTEM_ID;
        }

        if (sectionId.contains(MKTO_SYSTEM_ID + "_")) {
            return MKTO_SYSTEM_ID;
        }

        return "";
    }

    private boolean isSystemPresentInTemplate(String systemId) {
        return pomReader.getSourceSystem().contains(systemId) || pomReader.getDestinationSystem().contains(systemId);
    }
}
