package org.mule.core;

import com.google.common.base.Preconditions;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.app.event.*;
import org.apache.velocity.context.Context;
import org.apache.velocity.util.introspection.Info;
import org.mule.configuration.Configuration;
import org.mule.configuration.ConfigurationValidationProvider;
import org.mule.configuration.Placeholder;
import org.mule.configuration.Section;

import java.io.IOException;
import java.io.StringWriter;
import java.util.List;
import java.util.Scanner;

import static org.mule.configuration.ConfigurationValidationProvider.validateConfigurationValue;

/**
 * The {@link Engine} class handles the creation of a README.md file.
 * It does so based on configuration files provided or it uses the defaults.
 * Runs validations over the configurations and it also creates a table of content for the final
 * readme.
 *
 * @author damiansima
 */
public class Engine {
    private static final Logger log = Logger.getLogger(Engine.class);

    private static final String TEMPLATE_FILE_PATH = "README.template";

    private VelocityEngine ve;
    private Configuration config;
    private List<Section> sectionsToValidate;
    private List<Placeholder> placeholdersToValidate;


    private SectionSelector selector = new SectionSelector();

    public Engine() {
        this(ConfigurationValidationProvider.getPlaceholdersToValidate(), ConfigurationValidationProvider.getSectionsToValidate());
    }


    public SectionSelector getSelector() {
        return selector;
    }

    public void setSelector(SectionSelector selector) {
        this.selector = selector;
    }

    public Engine(List<Placeholder> placeholderstoValidate, List<Section> sectionsToValidate) {

        Preconditions.checkNotNull(sectionsToValidate, "The sections to validate should not be null.");
        Preconditions.checkNotNull(placeholderstoValidate, "The palceholders to validate should not be null.");

        this.sectionsToValidate = sectionsToValidate;
        this.placeholdersToValidate = placeholderstoValidate;

        ve = new VelocityEngine();
        ve.setProperty("resource.loader", "class");
        ve.setProperty("class.resource.loader.class", "org.apache.velocity.runtime.resource.loader.ClasspathResourceLoader");
        ve.setProperty("runtime.log.logsystem.class", "org.apache.velocity.runtime.log.NullLogSystem");

        ve.init();

        // TODO: the engine should have another constructor to parameterize files in the configuration
        // otherwise it can be tested
        this.config = new Configuration();
    }

    /**
     * The method makes use of it's internal parsers to get a hold of configurable parts of the
     * README file and finally build it.
     *
     * @return a string representing the README.md file
     * @throws IOException
     */
    public String buildReadme() throws IOException {
        validateSections();
        validatePlaceholders();

        return buildReadmeWithIndex(buildMainReadme());
    }

    /**
     * It verify that all the required place holders are contained in the configuration.
     */
    private void validatePlaceholders() {
        for (Placeholder ph : placeholdersToValidate) {
            if (ph.mandatory() && config.getPlaceholders().get(ph.placeholder()) == null) {
                throw new RuntimeException("The placeholder " + ph.placeholder() + " is mandatory but it has not been provided.");
            }
        }
    }

    /**
     * It verify that all the required sections are contained in the configuration.
     */
    private void validateSections() {
        for (Section s : sectionsToValidate) {
            if (s.mandatory() && config.getSections().get(s.section()) == null) {
                throw new RuntimeException("The section " + s.section() + " is mandatory but it has not been provided.");
            }
        }
    }

    private String buildMainReadme() {
        VelocityContext context = new VelocityContext();

        putSectionsIntoContext(context);
        putPlaceholdersIntoContext(context);

        Template template = ve.getTemplate(TEMPLATE_FILE_PATH);
        StringWriter writer = new StringWriter();

        template.merge(context, writer);


        return writer.toString();
    }

    private void putPlaceholdersIntoContext(VelocityContext context) {
        for (String key : config.getPlaceholders().keySet()) {
            context.put(key, validateConfigurationValue(key, config.getPlaceholders().get(key)));
        }
    }

    private void putSectionsIntoContext(VelocityContext context) {
        for (String key : config.getSections().keySet()) {
            if (selector.addSection(key)) {
                context.put(key, validateConfigurationValue(key, config.getSections().get(key)));
            }

        }
    }

    private String buildReadmeWithIndex(String readmeContent) {
        String index = buildIndex(readmeContent);

        return StringUtils.replace(readmeContent, "$index", index);
    }

    private String buildIndex(String readmeContent) {

        Scanner scanner = new Scanner(readmeContent);
        String carryRet = System.getProperty("line.separator");

        // skip first line as it's the title
        scanner.next();

        StringBuilder indexBuilder = new StringBuilder();
        while (scanner.hasNextLine()) {
            String line = scanner.nextLine();

            if (line.startsWith("#") && StringUtils.contains(line, "<a name=")) {
                String[] tokens = line.split("<a name=");

                String anchorText = tokens[0].replace("#", "");
                anchorText = anchorText.trim();

                String anchorLink = tokens[1].replace("\"", "").replace("/>", "").trim();

                String indexEntry = "[" + anchorText + "](#" + anchorLink + ")";
                if (line.startsWith("##")) {
                    indexEntry = "	* " + indexEntry;
                    indexBuilder.append(indexEntry).append(carryRet);
                } else if (line.startsWith("#")) {
                    indexEntry = "+ " + indexEntry;
                    indexBuilder.append(indexEntry).append(carryRet);
                }
            }
        }

        return indexBuilder.toString();
    }

}
