/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.functional;

import static org.mule.runtime.core.api.util.StreamingUtils.withCursoredEvent;
import static org.mule.test.allure.AllureConstants.XmlSdk.XML_SDK;

import static java.util.Arrays.asList;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;

import org.mule.extension.file.common.api.FileAttributes;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.connection.ConnectionValidationResult;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.runner.RunnerDelegateTo;

import java.util.Collection;
import java.util.Iterator;

import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runners.Parameterized;

import io.qameta.allure.Feature;

@Feature(XML_SDK)
@RunnerDelegateTo(Parameterized.class)
public class ModuleWithMultipleGlobalElementsTestCase extends AbstractCeXmlExtensionMuleArtifactFunctionalTestCase {

  private static final String SUB_DIRECTORY_NAME_A = "subDirectoryA";
  private static final String SUB_DIRECTORY_NAME_B = "subDirectoryB";

  @ClassRule
  public static TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Rule
  public SystemProperty workingDir = new SystemProperty("workingDir", temporaryFolder.getRoot().getAbsolutePath());

  @BeforeClass
  public static void setUp() throws Exception {
    if (!temporaryFolder.getRoot().exists()) {
      temporaryFolder.getRoot().mkdir();
    }
    temporaryFolder.newFolder(SUB_DIRECTORY_NAME_A);
    temporaryFolder.newFolder(SUB_DIRECTORY_NAME_B);
  }

  @AfterClass
  public static void tearDown() {
    temporaryFolder.delete();
  }

  @Parameterized.Parameter
  public String configFile;

  @Parameterized.Parameter(1)
  public String[] paths;

  @Parameterized.Parameter(2)
  public boolean shouldValidate;

  @Parameterized.Parameters(name = "{index}: Running tests for {0} ")
  public static Collection<Object[]> data() {
    return asList(scenarioWithoutTns(false),
                  scenarioWithoutTns(true),
                  scenarioWithTnsAndInternalOperations(false),
                  scenarioWithTnsAndInternalOperations(true),
                  // scenario with connection attributes (child element)
                  scenarioWithConnectionAttributes(false),
                  scenarioWithConnectionAttributes(true));
  }

  private static Object[] scenarioWithTnsAndInternalOperations(boolean shouldValidate) {
    return new Object[] {"flows/flows-using-module-calling-operations-within-module-with-global-elements.xml",
        new String[] {"modules/module-calling-operations-within-module-with-global-elements.xml"}, shouldValidate};
  }

  private static Object[] scenarioWithConnectionAttributes(boolean shouldValidate) {
    return new Object[] {"flows/flows-using-module-multiple-global-elements-connection.xml",
        new String[] {"modules/module-multiple-global-elements-connection.xml"}, shouldValidate};
  }

  private static Object[] scenarioWithoutTns(boolean shouldValidate) {
    return new Object[] {"flows/flows-using-module-multiple-global-elements.xml",
        new String[] {"modules/module-multiple-global-elements.xml"}, shouldValidate};
  }

  @Override
  protected boolean shouldValidateXml() {
    return shouldValidate;
  }

  @Override
  protected String[] getModulePaths() {
    return paths;
  }

  @Override
  protected String getConfigFile() {
    return configFile;
  }

  @Test
  public void testConnectionConfigPatternA() throws Exception {
    doTestConnection("configPatternA");
  }

  @Test
  public void testConnectionConfigPatternB() throws Exception {
    doTestConnection("configPatternB");
  }

  private void doTestConnection(String globalElement) throws MuleException {
    ConfigurationInstance config = extensionManager.getConfiguration(globalElement, testEvent());
    assertThat(config, is(notNullValue()));
    assertThat(config.getConnectionProvider().isPresent(), is(true));
    final ConnectionProvider connectionProvider = config.getConnectionProvider().get();
    final Object connect = connectionProvider.connect();
    final ConnectionValidationResult connectionValidationResult = connectionProvider.validate(connect);
    assertThat(connectionValidationResult.isValid(), is(true));
    connectionProvider.disconnect(connect);
  }

  @Test
  public void listPatternA() throws Exception {
    assertFlowResult("list-pattern-a", SUB_DIRECTORY_NAME_A);
  }

  @Test
  public void listPatternB() throws Exception {
    assertFlowResult("list-pattern-b", SUB_DIRECTORY_NAME_B);
  }

  private void assertFlowResult(String flowName, String subDirectoryName) throws Exception {
    withCursoredEvent(flowRunner(flowName).keepStreamsOpen().run(), event -> {

      Iterator<Message> messages = (Iterator<Message>) event.getMessage().getPayload().getValue();
      Message message = messages.next();
      assertThat(messages.hasNext(), is(false));
      FileAttributes attributes = (FileAttributes) message.getAttributes().getValue();
      assertThat(attributes.getName(), is(subDirectoryName));

      return event;
    });
  }

}
