/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.module.extension.oauth.clientcredentials;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.post;
import static com.github.tomakehurst.wiremock.client.WireMock.postRequestedFor;
import static com.github.tomakehurst.wiremock.client.WireMock.urlPathEqualTo;
import static com.github.tomakehurst.wiremock.client.WireMock.urlPathMatching;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mule.extension.http.api.HttpHeaders.Names.CONTENT_TYPE;
import static org.mule.oauth.client.api.state.ResourceOwnerOAuthContext.DEFAULT_RESOURCE_OWNER_ID;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.OK;
import static org.mule.test.oauth.TestOAuthExtension.TEST_OAUTH_EXTENSION_NAME;
import jakarta.inject.Inject;
import org.hamcrest.CoreMatchers;
import org.junit.Before;
import org.junit.Test;
import org.mule.oauth.client.api.state.ResourceOwnerOAuthContext;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.extension.api.client.ExtensionsClient;
import org.mule.runtime.extension.api.connectivity.oauth.ClientCredentialsState;
import org.mule.test.module.extension.oauth.BaseOAuthExtensionTestCase;
import org.mule.test.oauth.TestOAuthConnection;
import org.mule.test.oauth.TestOAuthConnectionState;

import java.util.Map;

public abstract class AbstractClientCredentialsExternalServerTestCase extends BaseOAuthExtensionTestCase {

  private LazyValue<ObjectStore> objectStore =
      new LazyValue<>(() -> muleContext.getObjectStoreManager().getObjectStore(CUSTOM_STORE_NAME));

  @Inject
  private ExtensionsClient extensionsClient;

  @Override
  protected void doSetUp() throws Exception {
    super.doSetUp();
    objectStore.get().clear();
    wireMock.stubFor(post(urlPathMatching("/" + TOKEN_PATH)).willReturn(aResponse()
        .withStatus(OK.getStatusCode())
        .withBody(accessTokenContent())
        .withHeader(CONTENT_TYPE, "application/json")));
  }

  @Override
  public boolean addToolingObjectsToRegistry() {
    return true;
  }

  @Before
  public void setOwnerId() throws Exception {
    ownerId = getCustomOwnerId();
  }

  @Test
  public void authenticate() throws Exception {
    TestOAuthConnectionState connection = ((TestOAuthConnection) flowRunner("getConnection")
        .run().getMessage().getPayload().getValue()).getState();

    assertConnectionState(connection);

    assertOAuthStateStored(CUSTOM_STORE_NAME, DEFAULT_RESOURCE_OWNER_ID);
  }

  @Test
  public void unauthorize() throws Exception {
    authenticate();

    flowRunner("unauthorize").run();
    ObjectStore objectStore = getObjectStore(CUSTOM_STORE_NAME);
    assertThat(objectStore.retrieveAll().size(), is(0));

    String refreshedToken = configureRefreshResponse();
    TestOAuthConnectionState connection = ((TestOAuthConnection) flowRunner("getConnection")
        .run().getMessage().getPayload().getValue()).getState();

    assertThat(connection.getState().getAccessToken(), CoreMatchers.equalTo(refreshedToken));
    assertRefreshToken(refreshedToken);
  }

  @Test
  public void refreshToken() throws Exception {
    authenticate();
    wireMock.resetAll();

    String refreshedToken = configureRefreshResponse();
    flowRunner("refreshToken").run();
    assertRefreshToken(refreshedToken);
  }

  @Test
  public void refreshTokenThroughExtensionClient() throws Exception {
    authenticate();
    wireMock.resetAll();

    String refreshedToken = configureRefreshResponse();

    extensionsClient.execute(TEST_OAUTH_EXTENSION_NAME,
                             "tokenExpired",
                             params -> params.withConfigRef("oauth"))
        .get();

    assertRefreshToken(refreshedToken);
  }

  @Test
  public void refreshTokenAsync() throws Exception {
    authenticate();
    wireMock.resetAll();

    String refreshedToken = configureRefreshResponse();
    flowRunner("refreshTokenAsync").run();
    assertRefreshToken(refreshedToken);
  }

  private String configureRefreshResponse() {
    String refreshedToken = ACCESS_TOKEN + "-refreshed";
    wireMock.stubFor(post(urlPathMatching("/" + TOKEN_PATH)).willReturn(aResponse()
        .withStatus(OK.getStatusCode())
        .withBody(accessTokenContent(refreshedToken))
        .withHeader(CONTENT_TYPE, "application/json")));
    return refreshedToken;
  }

  private void assertRefreshToken(String refreshedToken) throws Exception {
    wireMock.verify(postRequestedFor(urlPathEqualTo("/" + TOKEN_PATH)));

    Map entries = objectStore.get().retrieveAll();
    assertThat(entries.size(), is(1));
    ResourceOwnerOAuthContext context = (ResourceOwnerOAuthContext) entries.values().toArray()[0];
    assertThat(context.getAccessToken(), CoreMatchers.equalTo(refreshedToken));
  }

  @Override
  protected void assertAuthCodeState(TestOAuthConnectionState connection) {
    ClientCredentialsState state = (ClientCredentialsState) connection.getState();
    assertThat(state.getAccessToken(), is(ACCESS_TOKEN));
    assertThat(state.getExpiresIn().get(), is(EXPIRES_IN));
  }

}
