/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.config.ast;

import static org.mule.runtime.api.dsl.DslResolvingContext.getDefault;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_EXTRACT_CONNECTION_DATA;
import static org.mule.runtime.core.api.config.MuleProperties.MULE_BASE_DIRECTORY_PROPERTY;
import static org.mule.runtime.core.api.extension.provider.MuleExtensionModelProvider.getExtensionModel;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.extension.api.provider.RuntimeExtensionModelProviderLoaderUtils.discoverRuntimeExtensionModels;

import static java.nio.file.Files.readString;
import static java.util.Collections.singleton;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.io.FileMatchers.anExistingFile;

import org.mule.extension.db.internal.DbConnector;
import org.mule.extension.http.internal.temporary.HttpConnector;
import org.mule.extension.oauth2.OAuthExtension;
import org.mule.extension.socket.api.SocketsExtension;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.config.internal.context.ConnectionsDataExtractor;
import org.mule.runtime.module.extension.internal.loader.java.DefaultJavaExtensionModelLoader;
import org.mule.runtime.module.extension.internal.manager.DefaultExtensionManager;
import org.mule.tck.junit4.AbstractMuleContextTestCase;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.runner.infrastructure.ExtensionsTestInfrastructureDiscoverer;

import java.io.File;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class ConnectionsDataExtractorTestCase extends AbstractMuleContextTestCase {

  private static Set<ExtensionModel> runtimeExtensionModels;
  private DefaultExtensionManager extensionManager;

  @Rule
  public TemporaryFolder muleHome = new TemporaryFolder();

  @Rule
  public SystemProperty extractConnectionDataEnable = new SystemProperty(ENABLE_EXTRACT_CONNECTION_DATA, "true");

  @BeforeClass
  public static void beforeClass() {
    runtimeExtensionModels = discoverRuntimeExtensionModels();
  }

  @Before
  public void before() throws Exception {
    extensionManager = new DefaultExtensionManager();
    muleContext.setExtensionManager(extensionManager);
    initialiseIfNeeded(extensionManager, muleContext);
  }

  @Before
  public void setUpHome() {
    System.setProperty(MULE_BASE_DIRECTORY_PROPERTY, muleHome.getRoot().getAbsolutePath());
  }

  @After
  public void cleanUpHome() {
    System.clearProperty(MULE_BASE_DIRECTORY_PROPERTY);
  }

  @Test
  public void parametersTestHttpOauthProxyConfig() throws IOException {
    ArtifactAst ast = buildArtifactAst("parameters-test-http-oauth-proxy-config.xml",
                                       HttpConnector.class, SocketsExtension.class, OAuthExtension.class);

    new ConnectionsDataExtractor(ast).persist();

    final var report = new File(muleHome.getRoot().getAbsolutePath(), ".mule/.introspection/connections_data_artifact.csv");
    assertThat(report, anExistingFile());

    var reportContent = readString(report.toPath());
    assertThat(reportContent, is("""
        Component,Host,Port
        "httpRequestConfigWithOAuthProxyInline/connection(http:request-connection)","localhost","8081"
        "httpRequestConfigWithOAuthProxyInline/connection/0/0/0/0(http:proxy)","localhost","8083"
        "HTTP_Listener_config/connection(http:listener-connection)","localhost","8080"
        "HTTP_Listener_config/1/0/0/0(http:origin)","www.the-origin-of-time.com","80"
        "HTTP_Listener_config/1/0/0/1(http:origin)","www.the-origin-of-life.com","80"
        "defaultHttpListenerConfig/connection(http:listener-connection)","localhost","8081"
        "clientGlobalConfig/connection(http:request-connection)","localhost","8082"
        """));
  }

  @Test
  public void dbWithSpringDatasource() throws IOException {
    ArtifactAst ast = buildArtifactAst("db-with-spring-datasource.xml",
                                       DbConnector.class);

    new ConnectionsDataExtractor(ast).persist();

    final var report = new File(muleHome.getRoot().getAbsolutePath(), ".mule/.introspection/connections_data_artifact.csv");
    assertThat(report, anExistingFile());

    var reportContent = readString(report.toPath());
    assertThat(reportContent, is("""
        Component,Host,Port
        "pooledJdbcConfig/connection(db:data-source-connection)","","-1"
        """));
  }

  protected ArtifactAst buildArtifactAst(final String configFile, final Class... extensions) {
    ExtensionsTestInfrastructureDiscoverer discoverer = new ExtensionsTestInfrastructureDiscoverer(extensionManager);

    DefaultJavaExtensionModelLoader extensionModelLoader = new DefaultJavaExtensionModelLoader();
    Set<ExtensionModel> dependencies = new HashSet<>(singleton(getExtensionModel()));
    for (Class<?> annotatedClass : extensions) {
      dependencies.add(discoverer.discoverExtension(annotatedClass, extensionModelLoader, getDefault(dependencies)));
    }

    ArtifactAst parsedAst = AstXmlParser.builder()
        .withExtensionModels(muleContext.getExtensionManager().getExtensions())
        .withExtensionModels(runtimeExtensionModels)
        .withSchemaValidationsDisabled()
        .build()
        .parse(this.getClass().getClassLoader().getResource("ast/" + configFile));

    return parsedAst;
  }

}
