/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.integration.exceptions;

import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;
import static org.mule.runtime.http.api.HttpConstants.HttpStatus.INTERNAL_SERVER_ERROR;
import static org.mule.runtime.http.api.HttpConstants.Method.GET;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.core.api.Event;
import org.mule.runtime.core.api.processor.Processor;
import org.mule.runtime.http.api.HttpService;
import org.mule.runtime.http.api.domain.message.request.HttpRequest;
import org.mule.runtime.http.api.domain.message.response.HttpResponse;
import org.mule.service.http.TestHttpClient;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.test.AbstractIntegrationTestCase;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;

import java.util.concurrent.atomic.AtomicInteger;

public class ExceptionHandlingSourceTestCase extends AbstractIntegrationTestCase {

  @Rule
  public DynamicPort port = new DynamicPort("port");

  @Rule
  public TestHttpClient httpClient = new TestHttpClient.Builder(getService(HttpService.class)).build();

  @Override
  protected String getConfigFile() {
    return "org/mule/test/integration/exceptions/exception-handling-source-test.xml";
  }

  @Before
  public void before() {
    OnErrorCounterProcessor.count.set(0);
  }

  @Test
  public void errorSendingResponse() throws Exception {
    HttpRequest request = HttpRequest.builder()
        .setUri("http://localhost:" + port.getNumber() + "/errorSendingResponse")
        .setMethod(GET).build();

    HttpResponse response = httpClient.send(request, RECEIVE_TIMEOUT, false, null);

    assertThat(response.getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));
    assertThat(OnErrorCounterProcessor.count.get(), is(1));
  }

  @Test
  public void errorSendingErrorResponse() throws Exception {
    HttpRequest request = HttpRequest.builder()
        .setUri("http://localhost:" + port.getNumber() + "/errorSendingErrorResponse")
        .setMethod(GET).build();

    HttpResponse response = httpClient.send(request, RECEIVE_TIMEOUT, false, null);

    assertThat(response.getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));
    assertThat(OnErrorCounterProcessor.count.get(), is(1));
  }

  @Test
  public void errorSendingThrownError() throws Exception {
    HttpRequest request = HttpRequest.builder()
        .setUri("http://localhost:" + port.getNumber() + "/errorSendingThrownError")
        .setMethod(GET).build();

    HttpResponse response = httpClient.send(request, RECEIVE_TIMEOUT, false, null);

    assertThat(response.getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));
    assertThat(OnErrorCounterProcessor.count.get(), is(0));
  }

  @Test
  public void errorSendingErrorFromHandler() throws Exception {
    HttpRequest request = HttpRequest.builder()
        .setUri("http://localhost:" + port.getNumber() + "/errorSendingErrorFromHandler")
        .setMethod(GET).build();

    HttpResponse response = httpClient.send(request, RECEIVE_TIMEOUT, false, null);

    assertThat(response.getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));
    assertThat(OnErrorCounterProcessor.count.get(), is(1));
  }

  @Test
  public void errorSendingPropagatedError() throws Exception {
    HttpRequest request = HttpRequest.builder()
        .setUri("http://localhost:" + port.getNumber() + "/errorSendingPropagatedError")
        .setMethod(GET).build();

    HttpResponse response = httpClient.send(request, RECEIVE_TIMEOUT, false, null);

    assertThat(response.getStatusCode(), is(INTERNAL_SERVER_ERROR.getStatusCode()));
    assertThat(OnErrorCounterProcessor.count.get(), is(1));
  }

  public static class OnErrorCounterProcessor implements Processor {

    private static AtomicInteger count = new AtomicInteger(0);

    @Override
    public Event process(Event event) throws MuleException {
      count.incrementAndGet();
      return event;
    }
  }
}
