/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tls;

import static org.mule.tls.fips.DefaultTestConfiguration.getDefaultEnvironmentConfiguration;

import static org.apache.commons.io.FileUtils.getFile;
import static org.apache.http.impl.client.HttpClients.custom;

import static org.hamcrest.Matchers.anyOf;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;

import javax.net.ssl.SSLContext;

import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.ssl.SSLContexts;
import org.hamcrest.Matcher;

public class TlsTestUtils {

  // Expected validation error message for JDK 1.8.0_262.
  public static final String J8_262_SSL_ERROR_RESPONSE = "General SSLEngine problem";
  // Expected validation error message for JDK 1.8.0_275.
  public static final String J8_275_SSL_ERROR_RESPONSE = "Certificate signature validation failed";
  // Expected validation error message for JDK 11.
  public static final String J11_SSL_ERROR_RESPONSE = "PKIX path building failed";
  // Expected validation error message for JDK 17.
  public static final String J17_SSL_ERROR_RESPONSE = "PKIX path validation failed";
  // Expected validation messages for certificates when testing fips.
  public static final String BOUNCY_CASTLE_CERTIFICATE_UNKNOWN_ERROR_MESSAGE = "certificate_unknown";
  // Expected validation error message when using Netty.
  protected static final String NETTY_SSL_ERROR_RESPONSE = "Received fatal alert: handshake_failure";
  protected static final String APPROPRIATE_PROTOCOL_ERROR =
      "No appropriate protocol (protocol is disabled or cipher suites are inappropriate)";
  protected static final String NETTY_APPROPRIATE_PROTOCOL_ERROR = "Received fatal alert: protocol_version";

  private static final String keyPassword = "changeit";
  private static final String protocol = "TLSv1.2";

  public static HttpResponse executeGetRequest(String url) throws IOException, GeneralSecurityException, URISyntaxException {
    HttpClient client = getSecureClient();
    HttpGet getMethod = new HttpGet(url);
    return client.execute(getMethod);
  }

  private static HttpClient getSecureClient() throws IOException, GeneralSecurityException, URISyntaxException {
    HttpClient secureClient;
    secureClient = custom()
        .setSslcontext(getSslContext())
        .build();
    return secureClient;
  }

  private static SSLContext getSslContext() throws IOException, GeneralSecurityException, URISyntaxException {
    SSLContext customSslContext;
    File keyStore = new File(TlsTestUtils.class.getClassLoader().getResource(getDefaultEnvironmentConfiguration()
        .getTestSslKeyStore()).toURI());
    File trustStore = getFile(new File(TlsTestUtils.class.getClassLoader().getResource(getDefaultEnvironmentConfiguration()
        .getTestSslCaCerts()).toURI()));
    char[] storePass = getDefaultEnvironmentConfiguration().getTestStorePassword().toCharArray();
    char[] keyPass = keyPassword.toCharArray();
    customSslContext =
        SSLContexts.custom()
            .setKeyStoreType(getDefaultEnvironmentConfiguration().getTestStoreType())
            .useProtocol(protocol)
            .loadKeyMaterial(keyStore, storePass, keyPass)
            .loadTrustMaterial(trustStore, storePass)
            .build();
    return customSslContext;
  }

  public static Matcher<String> sslValidationError() {
    return is(anyOf(containsString(J8_262_SSL_ERROR_RESPONSE),
                    containsString(J8_275_SSL_ERROR_RESPONSE),
                    containsString(J11_SSL_ERROR_RESPONSE),
                    containsString(J17_SSL_ERROR_RESPONSE),
                    containsString(NETTY_SSL_ERROR_RESPONSE),
                    containsString(APPROPRIATE_PROTOCOL_ERROR),
                    containsString(NETTY_APPROPRIATE_PROTOCOL_ERROR)));
  }

}
