/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.integration.kerberos;

import static java.nio.file.Files.copy;
import static java.nio.file.Files.deleteIfExists;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.absent;
import static com.github.tomakehurst.wiremock.client.WireMock.any;
import static com.github.tomakehurst.wiremock.client.WireMock.anyUrl;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.getRequestedFor;
import static com.github.tomakehurst.wiremock.client.WireMock.matching;
import static com.github.tomakehurst.wiremock.client.WireMock.urlPathEqualTo;
import static com.github.tomakehurst.wiremock.client.WireMock.urlPathMatching;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;

import org.mule.functional.api.flow.FlowRunner;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.test.AbstractIntegrationTestCase;

import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.Map;

import com.github.tomakehurst.wiremock.core.WireMockConfiguration;
import com.github.tomakehurst.wiremock.junit.WireMockRule;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;

/**
 * Tests Kerberos proxy authentication: XML configuration validation and end-to-end HTTP flow with embedded KDC.
 */
public class KerberosProxyIntegrationTestCase extends AbstractIntegrationTestCase {

  private static KerberosTestInfrastructure kdcInfrastructure;

  @Rule
  public DynamicPort proxyPort = new DynamicPort("proxyPort");

  @Rule
  public DynamicPort httpPort = new DynamicPort("httpPort");

  @Rule
  public WireMockRule mockProxyServer = new WireMockRule(WireMockConfiguration.options()
      .bindAddress("127.0.0.1")
      .port(proxyPort.getNumber()));

  @Rule
  public WireMockRule mockTargetServer = new WireMockRule(WireMockConfiguration.options()
      .bindAddress("127.0.0.1")
      .port(httpPort.getNumber()));

  @BeforeClass
  public static void setUpKerberosInfrastructure() throws Exception {
    kdcInfrastructure = new KerberosTestInfrastructure();
    kdcInfrastructure.start();

    copy(kdcInfrastructure.getClientKeytab().toPath(),
         Paths.get("/tmp/client.keytab"),
         StandardCopyOption.REPLACE_EXISTING);
  }

  @Before
  public void setUpMockStubs() {
    // ensure clean state per test (clear requests and stubs)
    mockProxyServer.resetAll();
    mockTargetServer.resetAll();

    mockProxyServer.stubFor(any(urlPathMatching("/.*"))
        .withHeader("Proxy-Authorization", absent())
        .willReturn(aResponse()
            .withStatus(407)
            .withHeader("Proxy-Authenticate", "Negotiate")
            .withHeader("Content-Type", "text/plain")
            .withBody("Proxy Authentication Required")));

    mockProxyServer.stubFor(any(urlPathMatching("/.*"))
        .withHeader("Proxy-Authorization", matching("Negotiate [A-Za-z0-9+/=]+"))
        // forward authenticated requests to the actual target server
        .willReturn(aResponse()
            .proxiedFrom("http://localhost:" + httpPort.getNumber())));

    mockTargetServer.stubFor(get(urlPathEqualTo("/test"))
        .willReturn(aResponse().withStatus(200).withHeader("Content-Type", "text/plain")));
  }

  @AfterClass
  public static void tearDownKerberosInfrastructure() throws Exception {
    if (kdcInfrastructure != null) {
      kdcInfrastructure.stop();
      deleteIfExists(Paths.get("/tmp/client.keytab"));
    }
  }

  @Override
  protected String getConfigFile() {
    return "org/mule/test/integration/kerberos/kerberos-proxy-authentication-config.xml";
  }

  /**
   * Tests end-to-end Kerberos proxy authentication through Mule's HTTP client (non-preemptive by default). Verifies 407 challenge
   * on first request, then authenticated retry with Negotiate forwarded to the target.
   */
  @Test
  public void testEndToEndKerberosProxyAuthenticationFlow() throws Exception {
    // non-preemptive flow
    FlowRunner kerberosFlow = flowRunner("testKerberosProxyAuthenticationFlow");
    CoreEvent result = kerberosFlow.run();

    assertThat(result, is(notNullValue()));

    @SuppressWarnings("unchecked")
    Map<String, Object> responseMap = (Map<String, Object>) result.getMessage().getPayload().getValue();

    assertThat(responseMap.get("status"), is("success"));
    assertThat(responseMap.get("authentication"), is("kerberos_proxy"));
    assertThat(responseMap.get("response_status"), is(200));

    // first request without header (407), then retry with Negotiate
    mockProxyServer.verify(1, getRequestedFor(anyUrl())
        .withHeader("Proxy-Authorization", absent()));

    mockProxyServer.verify(1, getRequestedFor(anyUrl())
        .withHeader("Proxy-Authorization", matching("Negotiate [A-Za-z0-9+/=]+")));

    // exactly 2 requests to the proxy for this flow
    mockProxyServer.verify(2, getRequestedFor(anyUrl()));

    // verify the authenticated request reached the target server via the proxy
    mockTargetServer.verify(1, getRequestedFor(urlPathEqualTo("/test")));
    assertThat(responseMap.get("response_headers"), is(notNullValue()));
  }

  @Test
  public void testKerberosProxyAuthenticationFlowPreemptive() throws Exception {
    // preemptive flow
    FlowRunner kerberosFlow = flowRunner("testKerberosProxyAuthenticationFlowPreemptive");
    CoreEvent result = kerberosFlow.run();

    assertThat(result, is(notNullValue()));

    @SuppressWarnings("unchecked")
    Map<String, Object> responseMap = (Map<String, Object>) result.getMessage().getPayload().getValue();

    assertThat(responseMap.get("status"), is("success"));
    assertThat(responseMap.get("authentication"), is("kerberos_proxy_preemptive"));
    assertThat(responseMap.get("response_status"), is(200));

    // first request carries Negotiate header
    mockProxyServer.verify(1, getRequestedFor(anyUrl())
        .withHeader("Proxy-Authorization", matching("Negotiate [A-Za-z0-9+/=]+")));

    // only one request needed in preemptive mode
    mockProxyServer.verify(1, getRequestedFor(anyUrl()));

    // target reached
    mockTargetServer.verify(1, getRequestedFor(urlPathEqualTo("/test")));

    assertThat(responseMap.get("response_headers"), is(notNullValue()));
  }
}
