/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tls.sni;


import static org.mule.tck.util.FileUtils.getClassPathRoot;
import static org.mule.test.allure.AllureConstants.TlsSsl.TLS_SSL_FEATURE;
import static org.mule.tls.fips.DefaultTestConfiguration.getDefaultEnvironmentConfiguration;

import static java.nio.charset.StandardCharsets.UTF_8;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.Is.is;

import static org.junit.Assert.assertThrows;
import static org.junit.Assume.assumeThat;
import static org.junit.internal.matchers.ThrowableMessageMatcher.hasMessage;

import org.mule.runtime.http.api.HttpService;
import org.mule.service.http.TestHttpClient;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.AbstractIntegrationTestCase;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.InetAddress;
import java.net.URISyntaxException;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Feature;

@Feature(TLS_SSL_FEATURE)
public class HttpsRequesterSniTestCase extends AbstractIntegrationTestCase {

  private static final String FQDN = "localhost.localdomain";

  private static final String SERVER_PROTOCOL_ENABLED = "SSLv3,TLSv1,TLSv1.1,TLSv1.2";

  @Rule
  public TestHttpClient httpClient = new TestHttpClient.Builder(getService(HttpService.class)).build();

  @Rule
  public DynamicPort httpsPort = new DynamicPort("httpsPort");

  @Rule
  public SystemProperty password =
      new SystemProperty("password", getDefaultEnvironmentConfiguration().resolveStorePassword("changeit"));

  @Rule
  public SystemProperty sniClientTruststore =
      new SystemProperty("sniClientTruststore", getDefaultEnvironmentConfiguration().getTestSniClientTrustStore());

  @Rule
  public SystemProperty storeType = new SystemProperty("storeType", getDefaultEnvironmentConfiguration().getTestStoreType());

  private HttpsTestServerEnforcingSNI server;

  @Override
  protected String getConfigFile() {
    return "http-request-sni-config.xml";
  }

  @BeforeClass
  public static void createTlsPropertiesFile() throws Exception {
    PrintWriter writer = new PrintWriter(getTlsPropertiesFile(), UTF_8);
    writer.println("enabledProtocols=" + SERVER_PROTOCOL_ENABLED);
    writer.close();
  }

  @AfterClass
  public static void removeTlsPropertiesFile() {
    getTlsPropertiesFile().delete();
  }

  private static File getTlsPropertiesFile() {
    String path = System.getProperty("testClasspathDir");
    if (path == null) {
      path = getClassPathRoot(HttpsRequesterSniTestCase.class).getPath();
    }
    return new File(path, "tls-default.conf");
  }

  @Before
  public void prepareServer() throws IOException, URISyntaxException {
    server = new HttpsTestServerEnforcingSNI(httpsPort.getNumber());
    server.startServer();
  }

  @After
  public void teardownServer() {
    if (server != null) {
      server.stopServer();
    }
  }

  @Test
  public void testClientSNINotSentOnNonFQDN() {
    var flowRunner = flowRunner("requestFlowLocalhost").withPayload(TEST_MESSAGE);
    var error = assertThrows(Exception.class, flowRunner::run);
    assertThat(error, hasMessage(containsString("Remotely closed")));
  }

  @Test
  public void testClientSNISentOnFQDN() throws Exception {
    assumeFqdnWorksForThisTest(FQDN);

    flowRunner("requestFlowFQDN").withPayload(TEST_MESSAGE).run();
    assertThat(server.getHostname(), is(FQDN));
  }

  /*
   * SNI requires a fully qualified domain name. "localhost.localdomain" is used but it is not commonly present on MacOSX hosts.
   * An assumption will prevent its execution unless the domain exists. Although is recommended to add the aforementioned domain
   * to the /etc/ file if it's not present.
   */
  private static void assumeFqdnWorksForThisTest(String fqdn) {
    InetAddress address = null;
    try {
      address = InetAddress.getByName(fqdn);
    } catch (Exception ignored) {
    }

    assumeThat("SNI requires a fully qualified domain name. \"%s\" is used but it is not commonly present on MacOSX hosts. An assumption will prevent its execution unless the domain exists. Although is recommended to add the aforementioned domain to the /etc/hosts file if it's not present."
        .formatted(fqdn), address, is(notNullValue()));
  }
}
