/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tls.fips;

/**
 * Interface representing a configuration for testing. Provides methods to retrieve information about store types, SSL
 * certificates, key stores, and store passwords used during testing.
 */
public interface TestConfiguration {

  /**
   * Retrieves the key store type in PKCS#12 format.
   *
   * @return a {@code String} representing the PKCS#12 key store type.
   */
  String getKeyStorePKS12Type();

  /**
   * Retrieves the trust store type in JCEKS format.
   *
   * @return a {@code String} representing the JCEKS trust store type.
   */
  String getTrustStoreJCEKSType();

  /**
   * Resolves the password according to the environment.
   *
   * @param defaultPassword the default password
   * @return the resolvedPassword.
   */
  default String resolveStorePassword(String defaultPassword) {
    return defaultPassword;
  }

  /**
   * Retrieves the certificate authority entity.
   *
   * @return a {@code String} representing the certificate authority entity.
   */
  String getCertificateAuthorityEntity();

  /**
   * Retrieves the trust file used for Certificate Revocation List (CRL) validation.
   *
   * @return a {@code String} representing the trust file for CRL validation.
   */
  String getTrustFileForCrl();

  /**
   * Retrieves the trust store for the test client using Server Name Indication (SNI).
   *
   * @return a {@code String} representing the test SNI client trust store.
   */
  String getTestSniClientTrustStore();

  /**
   * Retrieves the type of the test store.
   *
   * @return a {@link String} representing the type of the test store.
   */
  String getTestStoreType();

  /**
   * Retrieves the CA certificates used for SSL during testing.
   *
   * @return a {@link String} representing the CA certificates for SSL.
   */
  String getTestSslCaCerts();

  /**
   * Retrieves the key store used for SSL during testing.
   *
   * @return a {@link String} representing the SSL key store.
   */
  String getTestSslKeyStore();

  /**
   * Retrieves the password for the test store.
   *
   * @return a {@link String} representing the test store password.
   */
  String getTestStorePassword();

  /**
   * Retrieves the SSL key store for testing, associated with a specific host name.
   *
   * @return a {@code String} representing the test SSL key store with the associated host name.
   */
  String getTestSslKeyStoreWithHostName();

  /**
   * Retrieves the server key store used for testing SSL connections.
   *
   * @return a {@link String} representing the server SSL key store.
   */
  String getTestServerKeyStore();

  /**
   * Retrieves the generic trust key store used for testing SSL connections.
   *
   * @return a {@link String} representing the generic trust SSL key store.
   */
  String getTestGenericTrustKeyStore();

  /**
   * Retrieves the key store for the test client.
   *
   * @return a {@code String} representing the test client's key store.
   */
  String getTestClientKeyStore();

  /**
   * Retrieves the key store for the TLS client that contains multiple keys.
   *
   * @return a {@code String} representing the TLS client's key store with multiple keys.
   */
  String getTlsClientKeyStoreWithMultipleKeys();

  /**
   * Retrieves the trust store file that excludes the Mule server certificate.
   *
   * @return a {@code String} representing the trust store file without the Mule server certificate.
   */
  String getTlsTrustStoreFileWithoutMuleServerCertificate();

  /**
   * Returns the cipher suite to be used for testing. A cipher suite defines the algorithms that will be used for encryption,
   * authentication, and key exchange during secure communication in tests.
   *
   * @return the name of the test cipher suite.
   */
  String getTestCipherSuite();

  /**
   * Returns an invalid or unsupported cipher suite for testing purposes. This can be used to simulate scenarios where an
   * incorrect or incompatible cipher suite is provided, helping to test error handling and fallback mechanisms.
   *
   * @return the name of the invalid test cipher suite.
   */
  String getInvalidTestKeyStore();

}

