/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tls;

import static org.mule.test.allure.AllureConstants.TlsSsl.TLS_SSL_FEATURE;
import static org.mule.tls.TlsTestUtils.BOUNCY_CASTLE_CERTIFICATE_UNKNOWN_ERROR_MESSAGE;
import static org.mule.tls.TlsTestUtils.J11_SSL_ERROR_RESPONSE;
import static org.mule.tls.TlsTestUtils.J17_SSL_ERROR_RESPONSE;
import static org.mule.tls.TlsTestUtils.J8_262_SSL_ERROR_RESPONSE;
import static org.mule.tls.TlsTestUtils.J8_275_SSL_ERROR_RESPONSE;
import static org.mule.tls.fips.DefaultTestConfiguration.getDefaultEnvironmentConfiguration;

import static org.hamcrest.CoreMatchers.anyOf;
import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;

import static org.junit.Assert.assertThrows;

import org.mule.functional.api.flow.FlowRunner;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.privileged.exception.MessagingException;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.AbstractIntegrationTestCase;

import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Feature;

@Feature(TLS_SSL_FEATURE)
public class TlsCustomConfigMultipleKeysTestCase extends AbstractIntegrationTestCase {

  @Rule
  public DynamicPort port = new DynamicPort("port");

  @Rule
  public SystemProperty serverKeyStoreType =
      new SystemProperty("storeType", getDefaultEnvironmentConfiguration().getTestStoreType());

  @Rule
  public SystemProperty tlsClientServerKeyStore =
      new SystemProperty("tlsClientServerKeyStore", getDefaultEnvironmentConfiguration().getTlsClientKeyStoreWithMultipleKeys());

  @Rule
  public SystemProperty tlsTrustStore =
      new SystemProperty("tlsTrustStore", getDefaultEnvironmentConfiguration().getTestGenericTrustKeyStore());

  @Rule
  public SystemProperty tlsTrustStoreFileWithoutMuleServerCertificate =
      new SystemProperty("tlsTrustStoreFileWithoutMuleServerCertificate",
                         getDefaultEnvironmentConfiguration().getTlsTrustStoreFileWithoutMuleServerCertificate());

  @Override
  protected String getConfigFile() {
    return "http-listener-custom-tls-multiple-keys-config.xml";
  }

  @Test
  public void acceptsConnectionWithValidCertificate() throws Exception {
    CoreEvent event = flowRunner("testFlowClientWithCertificate").withPayload(TEST_MESSAGE).run();
    assertThat(event.getMessage().getPayload().getValue(), equalTo(TEST_MESSAGE));
  }

  @Test
  public void rejectsConnectionWithInvalidCertificate() {
    FlowRunner flow = flowRunner("testFlowClientWithoutCertificate").withPayload(TEST_MESSAGE);
    MessagingException thrown = assertThrows(MessagingException.class, flow::run);
    assertThat(thrown.getMessage(), anyOf(containsString(J8_262_SSL_ERROR_RESPONSE),
                                          containsString(J8_275_SSL_ERROR_RESPONSE),
                                          containsString(J11_SSL_ERROR_RESPONSE),
                                          containsString(J17_SSL_ERROR_RESPONSE),
                                          containsString(BOUNCY_CASTLE_CERTIFICATE_UNKNOWN_ERROR_MESSAGE)));
  }

}
