/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package com.mycompany.log4j.appender;

import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.Core;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.Property;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Serializable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * A Log4j2 appender designed to test for deadlocks in logger context initialization.
 * On startup, it creates a logger in a separate thread which can trigger deadlocks in 
 * AbstractLoggerAdapter.getLoggersInContext() when addShutdownListener() is called.
 */
@Plugin(name = "DeadlockTestAppender", category = Core.CATEGORY_NAME, elementType = Appender.ELEMENT_TYPE, printObject = true)
public class DeadlockTestAppender extends AbstractAppender {

  private final String fileName;
  private final String prefix;

  protected DeadlockTestAppender(String name, Filter filter, Layout<? extends Serializable> layout,
                                 boolean ignoreExceptions, Property[] properties, String fileName, String prefix) {
    super(name, filter, layout, ignoreExceptions, properties);
    this.fileName = fileName;
    this.prefix = prefix;
  }

  @Override
  public void start() {
    super.start();

    // Getting a logger in a new thread during appender start can trigger a deadlock in 
    // AbstractLoggerAdapter.getLoggersInContext() when addShutdownListener() is called
    ExecutorService executor = Executors.newSingleThreadExecutor();
    try {
      Future<?> future = executor.submit(() -> {
        Logger logger = LoggerFactory.getLogger("org.mule.test.deadlock.appender.start");
        logger.info("DeadlockTestAppender started");
      });
      future.get(); // Wait for the thread to complete
    } catch (InterruptedException e) {
      Thread.currentThread().interrupt();
      LOGGER.error("Interrupted while waiting for logger initialization", e);
    } catch (ExecutionException e) {
      LOGGER.error("Error during logger initialization", e);
    } finally {
      executor.shutdown();
    }
  }

  @Override
  public void append(LogEvent event) {
    try {
      File file = new File(fileName);
      file.getParentFile().mkdirs();
      try (FileWriter writer = new FileWriter(file, true)) {
        String formattedMessage = prefix + " - " + event.getMessage().getFormattedMessage() + System.lineSeparator();
        writer.write(formattedMessage);
      }
    } catch (IOException e) {
      LOGGER.error("Error writing to deadlock test appender file", e);
    }
  }

  @PluginFactory
  public static DeadlockTestAppender createAppender(
      @PluginAttribute("name") String name,
      @PluginAttribute("fileName") String fileName,
      @PluginAttribute(value = "prefix", defaultString = "[DEADLOCK-TEST]") String prefix,
      @PluginElement("Layout") Layout<? extends Serializable> layout,
      @PluginElement("Filter") Filter filter) {

    if (name == null) {
      LOGGER.error("No name provided for DeadlockTestAppender");
      return null;
    }

    if (fileName == null) {
      LOGGER.error("No fileName provided for DeadlockTestAppender");
      return null;
    }

    return new DeadlockTestAppender(name, filter, layout, true, Property.EMPTY_ARRAY, fileName, prefix);
  }
}
