/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.test.integration.logging;

import static org.mule.tck.probe.PollingProber.probe;
import static org.mule.test.allure.AllureConstants.IntegrationTestsFeature.INTEGRATIONS_TESTS;
import static org.mule.test.allure.AllureConstants.Logging.LOGGING;
import static org.mule.test.allure.AllureConstants.Logging.LoggingStory.LOGGING_LIBS_SUPPORT;
import static org.mule.test.infrastructure.FileContainsInLine.hasLine;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import static org.hamcrest.CoreMatchers.containsString;

import org.mule.runtime.module.deployment.impl.internal.builder.ApplicationFileBuilder;
import org.mule.tck.util.CompilerUtils;
import org.mule.test.infrastructure.deployment.AbstractFakeMuleServerTestCase;

import java.io.File;

import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Features;
import io.qameta.allure.Issue;
import io.qameta.allure.Story;

/**
 * Tests for potential deadlocks in logger context initialization when multiple threads concurrently create loggers. This can
 * trigger deadlocks in AbstractLoggerAdapter.getLoggersInContext() when a logger context is accessed for the first time and
 * addShutdownListener() is called.
 */
@Features({@Feature(INTEGRATIONS_TESTS), @Feature(LOGGING)})
@Story(LOGGING_LIBS_SUPPORT)
@Issue("W-20419795")
public class Log4jLoggerContextDeadlockTestCase extends AbstractFakeMuleServerTestCase {

  public static final String APP_NAME = "logger-context-deadlock-app";

  @Rule
  public UseMuleLog4jContextFactory muleLogging = new UseMuleLog4jContextFactory();

  @Test
  @Description("Verifies that concurrent logger creation does not cause deadlocks in the logger context initialization")
  public void concurrentLoggerCreationDoesNotDeadlock() throws Exception {
    File deadlockTestAppenderClass =
        new CompilerUtils.SingleClassCompiler()
            .compile(new File(requireNonNull(requireNonNull(getClass().getClassLoader()
                .getResource("log/logger-context-deadlock/java/com/mycompany/log4j/appender/DeadlockTestAppender.java"))
                    .toURI())));

    final ApplicationFileBuilder deadlockTestAppBuilder =
        new ApplicationFileBuilder(APP_NAME).definedBy("log/logger-context-deadlock/mule/logger-context-deadlock-app.xml")
            .containingClass(deadlockTestAppenderClass, "com/mycompany/log4j/appender/DeadlockTestAppender.class")
            .usingResource("log/logger-context-deadlock/resources/log4j2.xml", "log4j2.xml");

    muleServer.start();
    muleServer.deploy(deadlockTestAppBuilder.getArtifactFile().toURI().toURL(), APP_NAME);

    File file = new File(muleServer.getLogsDir().toString() + "/logger-context-deadlock.log");

    String expectedPrefix = "[DEADLOCK-TEST]";
    String expectedMessage = "This message triggers concurrent logger creation";
    probe(() -> hasLine(containsString(expectedPrefix)).matches(file),
          () -> format("Custom prefix '%s' not present in the logs", expectedPrefix));
    probe(() -> hasLine(containsString(expectedMessage)).matches(file),
          () -> format("Expected message '%s' not present in the logs", expectedMessage));
  }
}

