/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.integration.singleapp;

import static org.mule.runtime.api.util.MuleSystemProperties.SINGLE_APP_MODE_PROPERTY;
import static org.mule.tck.probe.PollingProber.probe;
import static org.mule.test.allure.AllureConstants.ArtifactDeploymentFeature.SingleAppDeploymentStory.SINGLE_APP_DEPLOYMENT;
import static org.mule.test.allure.AllureConstants.IntegrationTestsFeature.INTEGRATIONS_TESTS;

import static java.lang.Boolean.TRUE;
import static java.lang.Thread.sleep;
import static java.nio.file.Files.copy;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;

import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.module.deployment.impl.internal.builder.ApplicationFileBuilder;
import org.mule.tck.junit4.rule.SystemProperty;
import org.mule.test.infrastructure.deployment.AbstractFakeMuleServerTestCase;

import org.junit.Rule;
import org.junit.Test;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature(INTEGRATIONS_TESTS)
@Story(SINGLE_APP_DEPLOYMENT)
public class SingleAppModeTestCase extends AbstractFakeMuleServerTestCase {

  private static final String SIMPLE_APP_NAME = "simple-app";
  private static final String SECOND_SIMPLE_APP_NAME = "second-simple-app";

  private static final String SIMPLE_APP_NAME_APP_DEPLOYMENT = "simple-app-1.0.0-mule-application";
  private static final String SIMPLE_APP_ROUTE = "apps/simple-app/simple-app.xml";
  private static final String SIMPLE_APP_LOG_CONF_ROUTE = "apps/simple-app/log4j2.xml";

  private static final long TIMEOUT = 5000;
  private static final long POLLING_INTERVAL = 1000;

  @Rule
  public SystemProperty singleAppMode = new SystemProperty(SINGLE_APP_MODE_PROPERTY, TRUE.toString());

  @Test
  @Description("An app can be deployed through the deployment service")
  public void oneAppCanBeDeployedThroughDeploymentService() throws Exception {
    muleServer.start();
    muleServer.deploy(new ApplicationFileBuilder(SIMPLE_APP_NAME).definedBy(SIMPLE_APP_ROUTE)
        .containingResource(SIMPLE_APP_LOG_CONF_ROUTE, "log4j2.xml").getArtifactFile().toUri().toURL(), SIMPLE_APP_NAME);

    assertThat(muleServer.findApplication(SIMPLE_APP_NAME), is(notNullValue()));
  }

  @Test
  @Description("When a file is placed in the apps folder it will be deployed")
  public void oneAppCanBeDeployedByPlacingItInTheAppsFolder() throws Exception {
    final var artifactFile = new ApplicationFileBuilder(SIMPLE_APP_NAME).definedBy(SIMPLE_APP_ROUTE)
        .containingResource(SIMPLE_APP_LOG_CONF_ROUTE, "log4j2.xml").getArtifactFile();

    copy(artifactFile, muleServer.getAppsDir().toPath().resolve(artifactFile.getFileName()));
    muleServer.start();
    probe(TIMEOUT, POLLING_INTERVAL, () -> muleServer.findApplication(SIMPLE_APP_NAME_APP_DEPLOYMENT) != null);
  }

  @Test
  @Description("When one app is deployed, placing a file in the /apps folder will have no effect.")
  public void oneAppCanBeDeployedByPlacingItInTheAppsFolderAndTheSecondAppIsNotDeployed() throws Exception {
    final var artifactFile = new ApplicationFileBuilder(SIMPLE_APP_NAME).definedBy(SIMPLE_APP_ROUTE)
        .containingResource(SIMPLE_APP_LOG_CONF_ROUTE, "log4j2.xml").getArtifactFile();
    copy(artifactFile,
         muleServer.getAppsDir().toPath().resolve(artifactFile.getFileName()));
    muleServer.start();
    probe(TIMEOUT, POLLING_INTERVAL, () -> muleServer.findApplication(SIMPLE_APP_NAME_APP_DEPLOYMENT) != null);

    final var artifactFile2 = new ApplicationFileBuilder(SECOND_SIMPLE_APP_NAME).definedBy(SIMPLE_APP_ROUTE)
        .containingResource(SIMPLE_APP_LOG_CONF_ROUTE, "log4j2.xml").getArtifactFile();
    copy(artifactFile2, muleServer.getAppsDir().toPath().resolve(artifactFile2.getFileName()));

    // Wait to verify that the app was not deployed.
    sleep(TIMEOUT);

    assertThat(muleServer.findApplication(SECOND_SIMPLE_APP_NAME), is(nullValue()));
  }

  @Test
  @Description("When one app is deployed, an attempt to deploy through the deployment service must fail")
  public void oneAppCanBeDeployedByPlacingItInTheAppsFolderAndTheSecondAppFailsToBeDeployedInDeploymentService()
      throws Exception {
    muleServer.start();

    final var appUrl = new ApplicationFileBuilder(SECOND_SIMPLE_APP_NAME).definedBy(SIMPLE_APP_ROUTE)
        .containingResource(SIMPLE_APP_LOG_CONF_ROUTE, "log4j2.xml").getArtifactFile().toUri().toURL();
    muleServer.deploy(appUrl, SIMPLE_APP_NAME);
    assertThat(muleServer.findApplication(SIMPLE_APP_NAME), is(notNullValue()));

    var thrown = assertThrows(AssertionError.class, () -> muleServer.deploy(appUrl, SECOND_SIMPLE_APP_NAME));
    assertThat(thrown.getMessage(), containsString("""
        Wanted but not invoked:
        deploymentListener.onDeploymentSuccess(
            "second-simple-app"
        );
        """));
  }
}
