/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.util;

import static java.util.Objects.requireNonNull;

import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;

/**
 * Extend the Apache Commons ClassUtils to provide additional functionality.
 * <p/>
 * <p>
 * This class is useful for loading resources and classes in a fault tolerant manner that works across different applications
 * servers. The resource and classloading methods are SecurityManager friendly.
 * </p>
 * Partially copied from mule-core to avoid visibility issues in tests.
 */
public class ClassUtils {

  /**
   * Executes the given {@code runnable} using the given {@code classLoader} as the current {@link Thread}'s context classloader.
   * <p>
   * This method guarantees that whatever the outcome, the thread's context classloader is set back to the value that it had
   * before executing this method
   *
   * @param classLoader the context {@link ClassLoader} on which the {@code runnable} should be executed
   * @param runnable    a closure
   */
  public static void withContextClassLoader(ClassLoader classLoader, Runnable runnable) {
    withContextClassLoader(classLoader, () -> {
      runnable.run();
      return null;
    });
  }

  /**
   * Executes the given {@code callable} using the given {@code classLoader} as the current {@link Thread}'s context classloader.
   * <p>
   * This method guarantees that whatever the outcome, the thread's context classloader is set back to the value that it had
   * before executing this method.
   * <p>
   * This method is only suitable for pieces that are only executed sporadically, are not executed at high concurrency rates and
   * are not critical performance wise. This is because all the lambdas and functional abstraction it relies on are really nice
   * from a coding POV, but it does add a performance overhead.
   *
   * @param classLoader the context {@link ClassLoader} on which the {@code runnable} should be executed
   * @param callable    a {@link Callable}
   * @return the value that the {@code callable} produced
   */
  public static <T> T withContextClassLoader(ClassLoader classLoader, Callable<T> callable) {
    final Thread currentThread = Thread.currentThread();
    final ClassLoader currentClassLoader = currentThread.getContextClassLoader();
    if (currentClassLoader != classLoader) {
      currentThread.setContextClassLoader(classLoader);
    }
    try {
      return callable.call();
    } catch (Exception e) {
      if (e instanceof RuntimeException) {
        throw (RuntimeException) e;
      }

      throw new RuntimeException(e);
    } finally {
      if (currentClassLoader != classLoader) {
        currentThread.setContextClassLoader(currentClassLoader);
      }
    }
  }

  /**
   * Sets {@code newClassLoader} as the context class loader for the {@code thread}, as long as said classloader is not the same
   * instance as {@code currentClassLoader}.
   * <p>
   * Since obtaining and setting the context classloader from a thread are expensive operations, the purpose of this method is to
   * avoid performing those operations when possible, which is why the two classloaders are tested not to be the same before
   * performing the set operation. For this method to make sense, {@code currentClassLoader} should actually be the current
   * context classloader from the {@code thread}.
   * <p>
   * This is how a typical use should look like:
   *
   * <pre>
   * Thread thread = Thread.currentThread();
   * ClassLoader currentClassLoader = thread.getContextClassLoader();
   * ClassLoader newClassLoader = getNewContextClassLoader(); // this one depends on your logic
   * ClassUtils.setContextClassLoader(thread, currentClassLoader, newClassLoader);
   * try {
   *   // execute your logic
   * } finally {
   *   // set things back as they were by reversing the arguments order
   *   ClassUtils.setContextClassLoader(thread, newClassLoader, currentClassLoader);
   * }
   * </pre>
   *
   * @param thread             the thread which context classloader is to be changed
   * @param currentClassLoader the thread's current context classloader
   * @param newClassLoader     the new classloader to be set
   * @since 4.3.0
   */
  public static void setContextClassLoader(Thread thread, ClassLoader currentClassLoader, ClassLoader newClassLoader) {
    if (currentClassLoader != newClassLoader) {
      thread.setContextClassLoader(newClassLoader);
    }
  }

  /**
   * Returns the list of interfaces implemented in a given class.
   *
   * @param aClass class to analyze. Non null.
   * @return the list of interfaces implemented in the provided class and all its super classes.
   */
  public static Class<?>[] findImplementedInterfaces(Class<?> aClass) {
    requireNonNull(aClass, "Class to analyze cannot be null");

    Class<?> currentClass = aClass;
    List<Class<?>> foundInterfaces = new LinkedList<>();
    while (currentClass != null) {
      Class<?>[] interfaces = currentClass.getInterfaces();
      Collections.addAll(foundInterfaces, interfaces);
      currentClass = currentClass.getSuperclass();
    }

    return foundInterfaces.toArray(new Class<?>[0]);
  }
}
