/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit5;

import static java.lang.Integer.getInteger;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.junit.jupiter.api.Assertions.assertThrows;

import java.io.IOException;
import java.net.ServerSocket;

import org.junit.jupiter.api.Test;

class DynamicPortTestCase {

  @DynamicPort(systemProperty = "my.dynamic.port.static")
  static Integer dynamicPortStatic;

  @DynamicPort(systemProperty = "my.dynamic.port.1")
  Integer dynamicPort1;

  @DynamicPort(systemProperty = "my.dynamic.port.2")
  Integer dynamicPort2;

  @Test
  void eachDynamicPortSetsADifferentProperty() {
    Integer systemProperty1 = getInteger("my.dynamic.port.1");
    assertThat(systemProperty1, is(dynamicPort1));

    Integer systemProperty2 = getInteger("my.dynamic.port.2");
    assertThat(systemProperty2, is(dynamicPort2));

    assertThat(dynamicPort1, is(not(dynamicPort2)));
  }

  @Test
  void dynamicPortCanBeBound() throws IOException {
    try (ServerSocket serverSocket = new ServerSocket(dynamicPort1)) {
      assertThat(serverSocket.getLocalPort(), is(dynamicPort1));
    }
  }

  @Test
  void dynamicPortCanNotBeBoundTwice() throws IOException {
    try (ServerSocket ignored = new ServerSocket(dynamicPort1)) {
      // A second socket on the same port can't be bound...
      var thrown = assertThrows(IOException.class, () -> new ServerSocket(dynamicPort1));
      assertThat(thrown.getMessage(), is("Address already in use"));
    }
  }

  @Test
  void propertyIsSetForStaticDynamicPort() {
    Integer systemProperty = getInteger("my.dynamic.port.static");
    assertThat(systemProperty, is(dynamicPortStatic));
  }
}
