/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.util;

import static org.apache.commons.io.FileUtils.listFiles;
import static org.apache.commons.io.IOUtils.closeQuietly;
import static org.apache.commons.io.IOUtils.copy;

import org.mule.runtime.core.api.util.compression.InvalidZipFileException;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.commons.io.filefilter.FalseFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.TrueFileFilter;

/**
 * <code>FileUtils</code> contains useful methods for dealing with files & directories. /*
 */
// @ThreadSafe
public class FileUtils {

  private static Collection<File> findFiles(File folder, IOFileFilter filter, boolean recursive) {
    return listFiles(folder, filter,
                     recursive ? TrueFileFilter.INSTANCE : FalseFileFilter.INSTANCE);
  }

  public static File findFileByName(File folder, final String filename) {
    Collection<File> files = findFiles(folder, new IOFileFilter() {

      @Override
      public boolean accept(File file) {
        return filename.equals(file.getName());
      }

      @Override
      public boolean accept(File dir, String name) {
        return true;
      }
    }, true);

    return files.isEmpty() ? null : files.iterator().next();
  }

  public static void unzip(File archive, File directory) throws IOException {
    ZipFile zip = null;

    if (directory.exists()) {
      if (!directory.isDirectory()) {
        throw new IOException("Directory is not a directory: " + directory);
      }
    } else {
      if (!directory.mkdirs()) {
        throw new IOException("Could not create directory: " + directory);
      }
    }

    try {
      zip = new ZipFile(archive);

      for (Enumeration<? extends ZipEntry> entries = zip.entries(); entries.hasMoreElements();) {
        ZipEntry entry = entries.nextElement();

        File file = verifyZipEntryPath(directory, entry);

        if (entry.isDirectory()) {
          if (!file.exists() && !file.mkdirs()) {
            throw new IOException("Could not create directory: " + file);
          }
        } else {
          if (!file.getParentFile().exists() && !file.getParentFile().mkdirs()) {
            throw new IOException("Unable to create folders for zip entry: " + entry.getName());
          }

          InputStream is = zip.getInputStream(entry);
          OutputStream os = new BufferedOutputStream(new FileOutputStream(file));
          try {
            copy(is, os);
          } finally {
            closeQuietly(is);
            closeQuietly(os);
          }
        }
      }
    } finally {
      if (zip != null) {
        zip.close();
      }
    }
  }

  private static File verifyZipEntryPath(File targetDirectory, ZipEntry entry) throws IOException {
    Path namePath = Paths.get(entry.getName());
    if (namePath.getRoot() != null) {
      // According to .ZIP File Format Specification (Section 4.4.17), the path can not be absolute
      throw new InvalidZipFileException("Absolute paths are not allowed: " + namePath.toString());
    }

    final File targetFile = new File(targetDirectory, entry.getName());
    // Not specified, but presents a security risk (allows overwriting external files)
    if (!targetFile.getCanonicalPath().startsWith(targetDirectory.getCanonicalPath())) {
      throw new InvalidZipFileException("External paths are not allowed: " + namePath.toString());
    }

    return targetFile;
  }

  private FileUtils() {
    // nothing to do
  }
}
