/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit5;

import static java.lang.reflect.Modifier.isStatic;

import static org.slf4j.LoggerFactory.getLogger;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;

import org.junit.jupiter.api.extension.AfterAllCallback;
import org.junit.jupiter.api.extension.AfterEachCallback;
import org.junit.jupiter.api.extension.BeforeAllCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.slf4j.Logger;

/**
 * A JUnit5 test extension that iterates over the class fields, and tries to inject the fields annotated with the annotation
 * {@link A}. The value to be injected is provided via the method {@link #createResource(A)}, and can be disposed with the method
 * {@link #disposeResource(A, V)}. Implementations have to provide those methods.
 *
 * @param <A> the annotation of the fields to be injected.
 * @param <V> the type of the annotated fields.
 */
public abstract class AbstractResourceExtension<A extends Annotation, V> implements
    BeforeEachCallback, AfterEachCallback,
    BeforeAllCallback, AfterAllCallback {

  private static final Logger LOGGER = getLogger(AbstractResourceExtension.class);

  private final Class<? extends Annotation> annotationClass;

  protected AbstractResourceExtension(Class<? extends A> annotationClass) {
    this.annotationClass = annotationClass;
  }

  protected abstract V createResource(A annotation);

  protected abstract void disposeResource(A annotation, V value);

  @Override
  public void beforeEach(ExtensionContext context) throws Exception {
    iterateAnnotatedFields(context, this::setField, false);
  }

  @Override
  public void afterEach(ExtensionContext context) throws Exception {
    iterateAnnotatedFields(context, this::unsetField, false);
  }

  @Override
  public void beforeAll(ExtensionContext context) throws Exception {
    iterateAnnotatedFields(context, this::setField, true);
  }

  @Override
  public void afterAll(ExtensionContext context) throws Exception {
    iterateAnnotatedFields(context, this::unsetField, true);
  }

  private void setField(Annotation annotation, Object instance, Field field) throws IllegalAccessException {
    LOGGER.debug("About to set field '{}' annotated with '{}'", field, annotation);
    field.set(instance, createResource((A) annotation));
  }

  private void unsetField(Annotation annotation, Object instance, Field field) throws IllegalAccessException {
    LOGGER.debug("About to clear field '{}' annotated with '{}'", field, annotation);
    disposeResource((A) annotation, (V) field.get(instance));
  }

  private void iterateAnnotatedFields(ExtensionContext context, TriConsumer<Annotation, Object, Field> callback,
                                      boolean processStatic)
      throws Exception {
    var testInstance = context.getTestInstance().orElse(null);
    var testClass = context.getRequiredTestClass();

    while (testClass != Object.class) {
      for (Field field : testClass.getDeclaredFields()) {
        var annotation = field.getAnnotation(annotationClass);
        if (annotation != null) {
          if (isStatic(field.getModifiers()) == processStatic) {
            field.setAccessible(true);
            callback.accept(annotation, testInstance, field);
          }
        }
      }
      testClass = testClass.getSuperclass();
    }
  }

  private interface TriConsumer<A, B, C> {

    void accept(A a, B b, C c) throws Exception;
  }
}
