/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tck.junit5;

import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;

import org.junit.jupiter.api.extension.AfterEachCallback;
import org.junit.jupiter.api.extension.BeforeEachCallback;
import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.ParameterContext;
import org.junit.jupiter.api.extension.ParameterResolver;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * JUnit 5 extension for managing system properties in tests. Can be used via {@link SystemProperty} annotation or registered
 * programmatically.
 */
public class SystemPropertyExtension implements BeforeEachCallback, AfterEachCallback, ParameterResolver {

  private final String propertyName;
  private final String propertyValue;

  private String previousValue;

  // For annotation-driven usage
  private static final Map<String, String> ANNOTATION_PROPERTIES = new ConcurrentHashMap<>();

  /**
   * No-args constructor for annotation-driven usage via @SystemProperty.
   */
  public SystemPropertyExtension() {
    this.propertyName = null;
    this.propertyValue = null;
  }

  /**
   * Constructor for programmatic usage via @RegisterExtension.
   */
  public SystemPropertyExtension(String propertyName, String propertyValue) {
    this.propertyName = propertyName;
    this.propertyValue = propertyValue;
  }

  private void set() {
    previousValue = getProperty(propertyName);
    setProperty(propertyName, propertyValue);
  }

  private void unSet() {
    if (previousValue == null) {
      clearProperty(propertyName);
    } else {
      setProperty(propertyName, previousValue);
    }
  }

  public String getPropertyValue() {
    return propertyValue;
  }

  @Override
  public String toString() {
    return "propertyName='" + propertyName + '\'' + ", propertyValue='" + propertyValue + '\'';
  }

  @Override
  public void beforeEach(ExtensionContext context) {
    // Annotation-driven usage
    if (propertyName == null) {
      processAnnotations(context);
    } else {
      // Programmatic usage
      this.set();
    }
  }

  @Override
  public void afterEach(ExtensionContext context) {
    // Annotation-driven usage
    if (propertyName == null) {
      clearAnnotations(context);
    } else {
      // Programmatic usage
      this.unSet();
    }
  }

  private void processAnnotations(ExtensionContext context) {
    SystemProperty[] annotations = findAnnotations(context);
    for (SystemProperty annotation : annotations) {
      String name = annotation.name();
      String value = annotation.value();
      String previousValue = getProperty(name);
      if (previousValue != null) {
        ANNOTATION_PROPERTIES.put(name, previousValue);
      }
      setProperty(name, value);
    }
  }

  private void clearAnnotations(ExtensionContext context) {
    SystemProperty[] annotations = findAnnotations(context);
    for (SystemProperty annotation : annotations) {
      String name = annotation.name();
      String previousValue = ANNOTATION_PROPERTIES.remove(name);
      if (previousValue != null) {
        setProperty(name, previousValue);
      } else {
        clearProperty(name);
      }
    }
  }

  private SystemProperty[] findAnnotations(ExtensionContext context) {
    return context.getRequiredTestClass().getAnnotationsByType(SystemProperty.class);
  }

  @Override
  public boolean supportsParameter(ParameterContext parameterContext, ExtensionContext extensionContext) {
    return false;
  }

  @Override
  public Object resolveParameter(ParameterContext parameterContext, ExtensionContext extensionContext) {
    return null;
  }
}
