/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tests.internal;

import static org.mule.runtime.core.api.util.UUID.getUUID;
import static org.mule.runtime.extension.api.annotation.param.MediaType.ANY;

import static java.lang.String.format;
import static java.lang.Thread.currentThread;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.route.Chain;

import org.slf4j.Logger;

public class AssertOperations {

  private static final Logger LOGGER = getLogger(AssertOperations.class);

  private static final ThreadLocal<String> taskTokenInThread = new ThreadLocal<>();

  /**
   * Verify that a thread switch on the nested {@code interceptedChain} did or did not occur, according to the given
   * {@code responseSameTask} flag.
   *
   * @param configuration
   * @param responseSameTask {@code true} in the intercepted chain must not do any thread switch, {@code false} if a thread switch
   *        is expected.
   * @param interceptedChain the chain in which a thread switch is expected or not.
   * @param callback
   */
  @MediaType(value = ANY, strict = false)
  public void assertIntercepting(@Optional(defaultValue = "true") boolean responseSameTask,
                                 Chain interceptedChain,
                                 CompletionCallback<Object, Object> callback) {

    String requestTaskToken;
    if (taskTokenInThread.get() != null) {
      requestTaskToken = taskTokenInThread.get();
    } else {
      requestTaskToken = generateTaskToken();
      taskTokenInThread.set(requestTaskToken);
    }

    interceptedChain.process(r -> {
      String responseTaskToken;
      if (taskTokenInThread.get() != null) {
        responseTaskToken = taskTokenInThread.get();
      } else {
        responseTaskToken = generateTaskToken();
      }

      if (responseSameTask && !requestTaskToken.equals(responseTaskToken)) {
        callback.error(new IllegalStateException(format("Response task (%s) was not same as request task (%s)",
                                                        responseTaskToken, requestTaskToken)));
      } else if (!responseSameTask && requestTaskToken.equals(responseTaskToken)) {
        final IllegalStateException e = new IllegalStateException(format("Response task (%s) was same as request task (%s)",
                                                                         responseTaskToken, requestTaskToken));
        LOGGER.error(e.getMessage(), e);
        callback.error(e);
      } else {
        callback.success(r);
      }
    }, (t, r) -> {
      callback.error(t);
    });
  }

  private String generateTaskToken() {
    return currentThread().getName() + " - " + getUUID();
  }


}
