/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static org.apache.commons.lang3.exception.ExceptionUtils.getStackTrace;
import org.mule.runtime.api.message.ErrorType;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;

/**
 * Immutable implementation of {@link ConnectionValidationResult}.
 *
 * @since 4.0
 */
public final class ImmutableConnectionValidationResult implements ConnectionValidationResult {

  private final boolean validationStatus;
  private final String message;
  private final ErrorType errorType;
  private final String exception;

  /**
   * Creates an instance of the connectivity testing result.
   *
   * @param validationStatus {@code true} if the connectivity testing was successful.
   * @param message {@link String} describing the reason why the connectivity testing failure.
   * @param errorType {@link ErrorType} the error type representing the connectivity testing failure.
   * @param exception {@link String} with the exception stack trace that cause the conectivity testing to fail.
   */
  private ImmutableConnectionValidationResult(boolean validationStatus, String message, ErrorType errorType,
                                              String exception) {
    this.validationStatus = validationStatus;
    this.message = message;
    this.errorType = errorType;
    this.exception = exception;
  }

  /**
   * @return {@link ImmutableConnectionValidationResult} successful.
   */
  public static ImmutableConnectionValidationResult success() {
    return new ImmutableConnectionValidationResult(true, null, null, null);
  }

  /**
   * @return {@link ImmutableConnectionValidationResult} failure with the error message and code.
   */
  public static ImmutableConnectionValidationResult failure(String message, ErrorType errorType, String exception) {
    return new ImmutableConnectionValidationResult(false, message, errorType, exception);
  }

  /**
   * @return {@link ImmutableConnectionValidationResult} failure with the error message and code.
   */
  public static ImmutableConnectionValidationResult failure(String message, ErrorType errorType, Throwable cause) {
    return new ImmutableConnectionValidationResult(false, message, errorType, getStackTrace(cause));
  }

  @Override
  public boolean isValid() {
    return this.validationStatus;
  }

  @Override
  public String getMessage() {
    return this.message;
  }

  @Override
  public ErrorType getErrorType() {
    return this.errorType;
  }

  @Override
  public String getException() {
    return this.exception;
  }

}
