/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client.exceptions;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.get;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorEntity;

import javax.ws.rs.core.Response;

/**
 * Exception for Tooling REST API, it includes as part of the message the {@link Response} body if there is one.
 *
 * @since 4.0
 */
public class ToolingServiceAPIException extends RuntimeException {

  private int statusCode;
  private String reasonPhrase;
  private ErrorEntity errorEntity;

  private ToolingServiceAPIException(int statusCode, String reasonPhrase, String causeMessage, Throwable cause) {
    super(createMessage(statusCode, reasonPhrase, causeMessage), cause);
    this.statusCode = statusCode;
    this.reasonPhrase = reasonPhrase;
  }

  private ToolingServiceAPIException(int statusCode, String reasonPhrase, ErrorEntity errorEntity, Throwable cause) {
    super(createMessage(statusCode, reasonPhrase, errorEntity.toString()), cause);
    this.statusCode = statusCode;
    this.reasonPhrase = reasonPhrase;
    this.errorEntity = errorEntity;
  }

  private static String createMessage(int statusCode, String reasonPhrase, String causeMessage) {
    return "HTTP " + statusCode + " " + reasonPhrase + (isNotBlank(causeMessage) ? ". " + causeMessage : "");
  }

  public int getStatusCode() {
    return this.statusCode;
  }

  public String getReasonPhrase() {
    return this.reasonPhrase;
  }

  public ErrorCode getErrorCode() {
    if (errorEntity == null) {
      return null;
    }
    return get(this.errorEntity.getErrorType());
  }

  /**
   * Creates a instance of the exception with the given response.
   *
   * @param response {@link Response} from server.
   */
  public static ToolingServiceAPIException newToolingServiceAPIException(Response response) {
    return newToolingServiceAPIException(response, null);
  }

  /**
   * Creates a instance of the exception with the given response and cause.
   *
   * @param response {@link Response} from server.
   * @param cause {@link Throwable} that caused the error.
   */
  public static ToolingServiceAPIException newToolingServiceAPIException(Response response, Throwable cause) {
    int statusCode = response.getStatusInfo().getStatusCode();
    String reasonPhrase = response.getStatusInfo().getReasonPhrase();

    if (response.hasEntity() && response.getMediaType().equals(APPLICATION_JSON_TYPE)) {
      ErrorEntity errorEntity = response.readEntity(ErrorEntity.class);
      return new ToolingServiceAPIException(statusCode, reasonPhrase, errorEntity, cause);
    } else {
      return new ToolingServiceAPIException(statusCode, reasonPhrase, response.readEntity(String.class), cause);
    }
  }
}
