/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static javax.ws.rs.client.ClientBuilder.newBuilder;
import static org.apache.commons.lang3.ClassUtils.getShortClassName;
import static org.mule.tooling.agent.rest.client.ConnectionValidationResultFactory.failure;
import static org.mule.tooling.agent.rest.client.ConnectionValidationResultFactory.success;
import static org.mule.tooling.agent.rest.client.api.ToolingServiceAPIClient.create;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.BUNDLE_NOT_FOUND;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.CONNECTIVITY_TESTING_OBJECT_NOT_FOUND;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.NO_SUCH_APPLICATION;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.UNKNOWN_ERROR;
import static org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode.UNSUPPORTED_CONNECTIVITY_TESTING_OBJECT;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.agent.rest.client.api.ToolingServiceAPIClient;
import org.mule.tooling.agent.rest.client.exceptions.ToolingServiceAPIException;
import org.mule.tooling.agent.rest.client.exceptions.model.ErrorCode;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationId.components.componentId.connection.model.ConnectivityTestingResponse;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationName.messageHistory.AgentTrackingNotificationResponse;
import org.mule.tooling.agent.rest.client.tooling.applications.model.ApplicationsPUTBody;
import org.mule.tooling.client.api.configuration.ssl.SslConfiguration;
import org.mule.tooling.client.api.connectivity.BundleNotFoundException;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingObjectNotFoundException;
import org.mule.tooling.client.api.connectivity.UnsupportedConnectivityTestingObjectException;
import org.mule.tooling.client.api.exception.DeploymentException;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.TimeoutException;
import org.mule.tooling.client.api.exception.ToolingException;

import com.mulesoft.agent.domain.tooling.BundleDescriptor;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewRequest;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewResponse;
import com.mulesoft.agent.external.handlers.tooling.TestConnectionRequest;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.URL;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.List;
import java.util.Optional;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import javax.ws.rs.ProcessingException;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * REST client for Mule Agent.
 *
 * @since 4.0
 */
public class RestAgentToolingService implements RuntimeToolingService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private ToolingServiceAPIClient client;
  private URL url;
  private int defaultReadTimeout;
  private static boolean VERBOSE_ERROR_ENABLED = true;
  private Optional<SSLContext> sslContext;

  /**
   * {@inheritDoc}
   */
  @Override
  public void setToolingApiUrl(URL url, long defaultConnectTimeout, long defaultReadTimeout,
                               Optional<SslConfiguration> sslConfiguration) {
    requireNonNull(url, "url cannot be null");

    if (logger.isDebugEnabled()) {
      logger.debug("Configuring using tooling API URL:{}, defaultConnectTimeout:{}ms, defaultReadTimeout:{}ms", url,
                   defaultConnectTimeout,
                   defaultReadTimeout);
    }
    this.sslContext = sslConfiguration.map(configuration -> {
      try {
        InputStream keyStoreFile = new FileInputStream(configuration.getKeyStoreConfig().getKeyStoreFile());
        InputStream trustStoreFile = new FileInputStream(configuration.getTrustStoreConfig().getTrustStoreFile());

        //TODO (gfernandes) this code should be also extensible by clients...
        KeyStore trustStore = KeyStore.getInstance("JKS");
        trustStore.load(trustStoreFile, configuration.getKeyStoreConfig().getKeyStorePassword().toCharArray());
        KeyStore keyStore = KeyStore.getInstance("JKS");
        keyStore.load(keyStoreFile, configuration.getTrustStoreConfig().getTrustStorePassword().toCharArray());

        SSLContext ctx = SSLContext.getInstance("TLSv1.2");
        KeyManagerFactory sunX509 = KeyManagerFactory.getInstance("SunX509");
        sunX509.init(keyStore, configuration.getKeyStoreConfig().getKeyStorePassword().toCharArray());
        TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance("SunX509", "SunJSSE");
        trustManagerFactory.init(trustStore);
        ctx.init(sunX509.getKeyManagers(), trustManagerFactory.getTrustManagers(), new SecureRandom());

        HttpsURLConnection.setDefaultHostnameVerifier((s, sslSession) -> true);
        return of(ctx);
      } catch (Exception e) {
        throw new IllegalArgumentException("Error while configuring mutual SSL to connect to the Mule Agent REST API.", e);
      }
    }).orElse(empty());
    this.defaultReadTimeout = Long.valueOf(defaultReadTimeout).intValue();
    this.client = create(url.toString(), Long.valueOf(defaultConnectTimeout).intValue(), this.defaultReadTimeout, sslContext);
    this.url = url;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isOperational() {
    ClientBuilder builder = newBuilder();
    sslContext.ifPresent(sslContext -> builder.sslContext(sslContext));
    Client client = builder.build();
    try {
      client.target(this.url.toString()).request().buildGet().invoke();
      return true;
    } catch (ProcessingException e) {
      if (e.getCause() instanceof ConnectException) {
        return false;
      }
      // In case of another error just propagate the exception
      throw new ToolingException("Error while checking if Mule Agent REST service is available", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployApplication(File appLocation) {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/applications appLocation:[{}]", appLocation);
    }
    try {
      return client.tooling.applications
          .put(new ApplicationsPUTBody(appLocation.getAbsolutePath()), VERBOSE_ERROR_ENABLED, defaultReadTimeout)
          .getApplicationId();
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new DeploymentException("Couldn't deploy the application", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String deployApplication(InputStream inputStream) throws DeploymentException, ServiceUnavailableException {
    logger.debug("PUT:tooling/applications appLocation");
    try {
      return client.tooling.applications.put(inputStream, VERBOSE_ERROR_ENABLED, defaultReadTimeout)
          .getApplicationId();
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new DeploymentException("Couldn't deploy the application", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disposeApplication(String applicationId) throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/applications/{}", applicationId);
    }
    try {
      client.tooling.applications.applicationId(applicationId).delete(VERBOSE_ERROR_ENABLED, defaultReadTimeout);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException(format("Couldn't dispose application, for applicationId: %s", applicationId),
                                 e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectionValidationResult testConnection(String applicationId, String componentId, long readTimeout)
      throws NoSuchApplicationException,
      UnsupportedConnectivityTestingObjectException, ConnectivityTestingObjectNotFoundException, ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/connection", applicationId, componentId);
    }
    try {
      ConnectivityTestingResponse response =
          client.tooling.applications.applicationId(applicationId).components.componentId(componentId).connection
              .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue());
      return response.getValidationStatus() ? success()
          : failure(response.getMessage(), response.getErrorType(), response.getException());
    } catch (ToolingServiceAPIException e) {
      return handleConnectivityTestingError(e);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while deploying application", e);
    }
  }

  @Override
  public ConnectionValidationResult testConnection(List<BundleDescriptor> dependencies, ArtifactDeclaration artifactDeclaration,
                                                   String componentId, long readTimeout)
      throws UnsupportedConnectivityTestingObjectException, ConnectivityTestingObjectNotFoundException,
      ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/components/{}/connection", componentId);
    }
    try {
      ConnectivityTestingResponse response = client.tooling.components.componentId(componentId).connection
          .put(createRequest(dependencies, artifactDeclaration), VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue());
      return response.getValidationStatus() ? success()
          : failure(response.getMessage(), response.getErrorType(), response.getException());
    } catch (ToolingServiceAPIException e) {
      return handleConnectivityTestingError(e);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while doing connectivity testing for a temporary application", e);
    }
  }

  @Override
  public PreviewResponse runDataWeave(String applicationId, PreviewRequest request) {
    if (logger.isDebugEnabled()) {
      logger.debug("POST:tooling/applications/{}/dataweave/execute", applicationId);
    }
    try {
      PreviewResponse response =
          client.tooling.applications.applicationId(applicationId).dataweave.execute.post(request, VERBOSE_ERROR_ENABLED,
                                                                                          defaultReadTimeout);
      return response;
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while executing dataweave script for a temporary application", e);
    }
  }

  private TestConnectionRequest createRequest(List<BundleDescriptor> dependencies, ArtifactDeclaration artifactDeclaration) {
    TestConnectionRequest request = new TestConnectionRequest();
    request.setArtifactDeclaration(artifactDeclaration);
    request.setDependencies(dependencies);
    return request;
  }

  private ConnectionValidationResult handleConnectivityTestingError(ToolingServiceAPIException e) {
    final ErrorCode errorCode = e.getErrorCode();
    if (errorCode == null) {
      throw new ToolingException("Internal server error while doing connectivity testing", e);
    }
    if (errorCode == NO_SUCH_APPLICATION) {
      throw new NoSuchApplicationException("Application resource not found", e);
    } else if (errorCode == BUNDLE_NOT_FOUND) {
      throw new BundleNotFoundException("Extension bundle not found", e);
    } else if (errorCode == CONNECTIVITY_TESTING_OBJECT_NOT_FOUND) {
      throw new ConnectivityTestingObjectNotFoundException("Component resource not found", e);
    } else if (errorCode == UNSUPPORTED_CONNECTIVITY_TESTING_OBJECT) {
      throw new UnsupportedConnectivityTestingObjectException("Component doesn't support connectivity testing",
                                                              e);
    } else if (errorCode == UNKNOWN_ERROR) {
      throw new ToolingException("Unknown error while doing connectivity testing", e);
    }
    throw new ToolingException("Internal server error while doing connectivity testing for application",
                               e);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(String applicationId, String componentLocation, long readTimeout)
      throws ServiceUnavailableException {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/keys", applicationId, componentLocation);
    }
    try {
      return client.tooling.applications.applicationId(applicationId).components.componentId(componentLocation).keys
          .get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue());
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while getting Metadata keys", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(String applicationId,
                                                                                          String componentLocation,
                                                                                          long readTimeout)
      throws ServiceUnavailableException {
    return getMetadata(applicationId, componentLocation, readTimeout);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(String applicationId,
                                                                                    String componentLocation,
                                                                                    long readTimeout)
      throws ServiceUnavailableException {
    return getMetadata(applicationId, componentLocation, readTimeout);
  }

  private <T extends ComponentModel> MetadataResult<ComponentMetadataDescriptor<T>> getMetadata(String applicationId,
                                                                                                String componentLocation,
                                                                                                long readTimeout) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/metadata", applicationId, componentLocation);
    }
    try {
      return client.tooling.applications.applicationId(applicationId).components
          .componentId(componentLocation).metadata.get(VERBOSE_ERROR_ENABLED, Long.valueOf(readTimeout).intValue());
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while getting Metadata", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void enableMessageHistory(String applicationName) throws NoSuchApplicationException {
    if (logger.isDebugEnabled()) {
      logger.debug("PUT:tooling/applications/{}/messagehistory/", applicationName);
    }
    try {
      client.tooling.applications.applicationName(applicationName).messageHistory
          .put(VERBOSE_ERROR_ENABLED, defaultReadTimeout);
    } catch (ToolingServiceAPIException e) {
      final ErrorCode errorCode = e.getErrorCode();
      if (errorCode == NO_SUCH_APPLICATION) {
        throw new NoSuchApplicationException("Application resource not found", e);
      }
      throw new ToolingException("Error while enabling application for message history", e);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while enabling application for message history", e);
    }
  }

  @Override
  public List<AgentTrackingNotificationResponse> consumeMessageHistoryNotifications(String applicationName, int chunkSize) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/messagehistory/", applicationName);
    }
    try {
      return client.tooling.applications.applicationName(applicationName).messageHistory.get(chunkSize, VERBOSE_ERROR_ENABLED,
                                                                                             defaultReadTimeout);
    } catch (ToolingServiceAPIException e) {
      final ErrorCode errorCode = e.getErrorCode();
      if (errorCode == NO_SUCH_APPLICATION) {
        throw new NoSuchApplicationException("Application resource not found", e);
      }
      throw new ToolingException("Error while getting application notifications for MessageHistory", e);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while getting application notifications for MessageHistory", e);
    }
  }

  @Override
  public void disableMessageHistory(String applicationName) {
    if (logger.isDebugEnabled()) {
      logger.debug("DELETE:tooling/applications/{}/messagehistory/", applicationName);
    }
    try {
      client.tooling.applications.applicationName(applicationName).messageHistory.delete(VERBOSE_ERROR_ENABLED,
                                                                                         defaultReadTimeout);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException("Error while disabling application for MessageHistory", e);
    }
  }

  @Override
  public ValueResult getValues(String applicationName, String location, String providerName) {
    if (logger.isDebugEnabled()) {
      logger.debug("GET:tooling/applications/{}/components/{}/valueProviders/{}", applicationName, location, providerName);
    }
    try {
      return client.tooling.applications.applicationId(applicationName).components.componentId(location).valueProviders
          .get(providerName, defaultReadTimeout);
    } catch (Exception e) {
      if (e.getCause() instanceof ConnectException) {
        throw new ServiceUnavailableException("Mule Agent REST service is not available", e);
      }
      if (e.getCause() instanceof SocketTimeoutException) {
        throw new TimeoutException("Mule Agent REST service timed out", e);
      }
      throw new ToolingException(format("Error while trying to resolve Values for the ValueProvider [%s] located in [%s] of the application with ID [%s]",
                                        providerName, location, applicationName),
                                 e);
    }
  }

  @Override
  public String toString() {
    return format("%s{url=%s}", getShortClassName(this.getClass()), url);
  }
}
