/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static java.util.Collections.singletonList;
import static org.apache.commons.lang3.tuple.ImmutablePair.of;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;
import org.mule.tck.junit4.rule.DynamicPort;
import org.mule.tooling.client.api.exception.DeploymentException;
import org.mule.tooling.client.test.RuntimeType;

import com.icegreen.greenmail.util.GreenMail;
import com.icegreen.greenmail.util.ServerSetup;

import java.io.File;
import java.io.FileInputStream;
import java.net.URISyntaxException;
import java.util.List;

import org.apache.commons.lang3.tuple.ImmutablePair;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;

@Feature("Remote Runtime Tooling Service")
@Story("DeployToolingArtifact")
public class DeployApplicationTestCase extends AbstractRestAgentToolingServiceTestCase {

  @Rule
  public DynamicPort emailServerPort = new DynamicPort("emailServerPort");
  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private GreenMail greenMail;

  public DeployApplicationTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Before
  public void setUpEmailServer() {
    ServerSetup serverSetup = new ServerSetup(emailServerPort.getNumber(), null, "pop3");
    greenMail = new GreenMail(serverSetup);
    greenMail.setUser("foo", "pwd");
    greenMail.start();
  }

  @After
  public void stopEmailServer() {
    if (greenMail != null) {
      greenMail.stop();
    }
  }

  @Override
  protected List<ImmutablePair<String, String>> getStartupSystemProperties() {
    return singletonList(of("emailServerPort", String.valueOf(emailServerPort.getNumber())));
  }

  @Test
  @Description("Checks Deployment of a Tooling Application from Mule Rutime Agent using the RuntimeToolingService")
  public void deployApplication() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    String applicationId = restAgentToolingService.deployApplication(emailAppLocation);

    assertThat(applicationId, is(notNullValue()));
  }

  @Test
  @Description("Checks Deployment of a Tooling Application using InputStream from Mule Rutime Agent using the RuntimeToolingService")
  public void deployApplicationInputStream() throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    File targetFile = temporaryFolder.newFile();
    compress(targetFile, emailAppLocation);

    String applicationId = restAgentToolingService.deployApplication(new FileInputStream(targetFile));

    assertThat(applicationId, is(notNullValue()));
  }

  @Test
  @Description("Checks Deployment of a Tooling Application with a non existing location from Mule Rutime Agent using the RuntimeToolingService")
  public void deployNonExistentApplicationLocation() throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/nonexistent");

    expectedException.expect(DeploymentException.class);
    expectedException.expectMessage(containsString("Couldn't deploy the application"));
    restAgentToolingService.deployApplication(emailAppLocation);
  }

  @Test
  @Description("Checks Deployment of a Tooling Application that contains a wrong configuration from Mule Rutime Agent using the RuntimeToolingService")
  public void deployWrongConfigShouldNotFail() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/wrong-email");

    restAgentToolingService.deployApplication(emailAppLocation);
  }

}
