/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;
import org.mule.tooling.client.api.exception.DeploymentException;
import org.mule.tooling.client.test.RuntimeType;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URISyntaxException;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

@Feature("Remote Runtime Tooling Service - Applications")
@Story("DeployToolingArtifact")
public class DeployApplicationTestCase extends AbstractDeploymentTestCase {

  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  public DeployApplicationTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Before
  public void beforeTest() {
    setUpEmailServer();
  }

  @After
  public void afterTest() {
    stopEmailServer();
  }

  @Test
  @Description("Checks Deployment of a Tooling Application from Mule Runtime Agent using the RuntimeToolingService")
  public void deployApplication() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    String applicationId = restAgentToolingService.deployApplication(emailAppLocation);

    assertThat(applicationId, is(notNullValue()));
  }

  @Test
  @Description("Checks Deployment of a Tooling Application that depends on a Domain from Mule Runtime Agent using the RuntimeToolingService")
  public void deployApplicationDeclaresDomainDependency() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain");

    expectedException.expect(DeploymentException.class);
    expectedException
        .expectMessage(containsString("Domain 'agent-domain-01-4.1.0-SNAPSHOT-mule-domain' has to be deployed in order to deploy Application"));
    restAgentToolingService.deployApplication(emailAppLocation);
  }

  @Test
  public void deployApplicationWithDomainExploded() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");
    String domainId = restAgentToolingService.deployDomain(emailDomainLocation);

    assertThat(domainId, is(notNullValue()));

    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain-exploded");
    String applicationId = restAgentToolingService.deployApplication(emailAppLocation, domainId);

    assertThat(applicationId, is(notNullValue()));

    restAgentToolingService.disposeApplication(applicationId);
    restAgentToolingService.disposeDomain(domainId);
  }

  @Test
  @Description("Checks Deployment of a Tooling Application using InputStream from Mule Runtime Agent using the RuntimeToolingService")
  public void deployApplicationInputStream() throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    File targetFile = temporaryFolder.newFile();
    compress(targetFile, emailAppLocation);

    String applicationId = restAgentToolingService.deployApplication(new FileInputStream(targetFile));

    assertThat(applicationId, is(notNullValue()));
  }

  @Test
  public void deployApplicationWithDomainExplodedUsingInputStream() throws URISyntaxException, IOException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");

    File domainTargetFile = temporaryFolder.newFile();
    compress(domainTargetFile, emailDomainLocation);

    String domainId = restAgentToolingService.deployDomain(new FileInputStream(domainTargetFile));

    assertThat(domainId, is(notNullValue()));

    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain-exploded");

    File appTargetFile = temporaryFolder.newFile();
    compress(appTargetFile, emailAppLocation);


    String applicationId = restAgentToolingService.deployApplication(new FileInputStream(appTargetFile), domainId);

    assertThat(applicationId, is(notNullValue()));

    restAgentToolingService.disposeApplication(applicationId);
    restAgentToolingService.disposeDomain(domainId);
  }

  @Test
  @Description("Checks Deployment of a Tooling Application with a non existing location from Mule Runtime Agent using the RuntimeToolingService")
  public void deployNonExistentApplicationLocation() throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/nonexistent");

    expectedException.expect(DeploymentException.class);
    expectedException.expectMessage(containsString("Couldn't deploy the application"));
    restAgentToolingService.deployApplication(emailAppLocation);
  }

  @Test
  @Description("Checks Deployment of a Tooling Application that contains a wrong configuration from Mule Runtime Agent using the RuntimeToolingService")
  public void deployWrongConfigShouldNotFail() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/wrong-email");

    restAgentToolingService.deployApplication(emailAppLocation);
  }

}
