/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client.api;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static jersey.repackaged.com.google.common.base.Preconditions.checkNotNull;
import static org.glassfish.jersey.client.ClientProperties.CONNECT_TIMEOUT;
import static org.glassfish.jersey.client.ClientProperties.READ_TIMEOUT;
import org.mule.tooling.agent.rest.client.tooling.Tooling;

import com.fasterxml.jackson.jaxrs.json.JacksonJaxbJsonProvider;

import java.util.Optional;

import javax.net.ssl.SSLContext;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;

import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.filter.LoggingFilter;

/**
 * REST client implementation for Tooling REST API from Mule Runtime Agent.
 *
 * @since 4.0
 */
public class ToolingServiceAPIClient {

  public final Tooling tooling;

  private ToolingServiceAPIClient(String toolingApiUrl, int defaultConnectTimeout, int defaultReadTimeout,
                                  Optional<SSLContext> sslContext) {
    checkNotNull(toolingApiUrl, "baseUrl cannot be null");

    tooling = new Tooling(toolingApiUrl, newClient(defaultConnectTimeout, defaultReadTimeout, sslContext));
  }

  private Client newClient(int connectTimeout, int readTimeout, Optional<SSLContext> sslContextOptional) {
    ClientConfig configuration = new ClientConfig(new JacksonJaxbJsonProvider().configure(FAIL_ON_UNKNOWN_PROPERTIES, false));
    configuration.property(CONNECT_TIMEOUT, connectTimeout);
    configuration.property(READ_TIMEOUT, readTimeout);
    ClientBuilder builder = ClientBuilder.newBuilder();

    builder.register(new LoggingFilter());
    builder.withConfig(configuration);

    sslContextOptional.ifPresent(sslContext -> builder.sslContext(sslContext));

    return builder.build();
  }

  /**
   * Factory method that creates the REST client.
   *
   * @param baseUrl               base URL for the service. Must not be {@code null}.
   * @param defaultConnectTimeout default connection timeout for REST client.
   * @param defaultReadTimeout    default read timeout for REST client.
   * @param sslContext            ssl configuration for the REST client.
   * @return {@link ToolingServiceAPIClient} to access REST operations.
   */
  public static ToolingServiceAPIClient create(String baseUrl, int defaultConnectTimeout, int defaultReadTimeout,
                                               Optional<SSLContext> sslContext) {
    return new ToolingServiceAPIClient(baseUrl, defaultConnectTimeout, defaultReadTimeout, sslContext);
  }

}
