/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static java.util.Collections.emptyMap;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingObjectNotFoundException;
import org.mule.tooling.client.test.RuntimeType;

import java.io.File;
import java.net.URISyntaxException;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;

@Feature("Remote Runtime Tooling Service")
@Story("ConnectivityTesting")
public class ConnectivityTestingTestCase extends AbstractDeploymentTestCase {

  @Rule
  public ExpectedException expectedException = none();

  public ConnectivityTestingTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Test
  @Description("Checks connectivity testing for a global element declared on Domain")
  public void connectivityTestingConfigDeclaredOnDomain() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File location = new File(targetTestClassesFolder, "domains/email-domain-exploded");

    String domainId = restAgentToolingService.deployDomain(location, getDeploymentProperties());
    setUpEmailServer();
    try {
      assertThat(domainId, is(notNullValue()));
      final ConnectionValidationResult result =
          restAgentToolingService.testConnectionDomain(domainId, "emailConfig", -1);
      assertThat(result.isValid(), is(true));
    } finally {
      restAgentToolingService.disposeDomain(domainId);
      stopEmailServer();
    }
  }


  @Test
  @Description("Checks connectivity testing for a global element declared on application works even when using a custom domain")
  public void connectivityTestingConfigDeclaredOnApplication() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");
    String domainId = restAgentToolingService.deployDomain(emailDomainLocation, getDeploymentProperties());

    assertThat(domainId, is(notNullValue()));

    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain-exploded");
    String applicationId = restAgentToolingService.deployApplication(emailAppLocation, domainId, getDeploymentProperties());

    assertThat(applicationId, is(notNullValue()));

    try {
      assertThat(applicationId, is(notNullValue()));

      expectedException.expect(ConnectivityTestingObjectNotFoundException.class);
      expectedException.expectMessage("Component resource not found");
      restAgentToolingService.testConnectionApplication(applicationId, "httpListenerConfig", -1);
    } finally {
      restAgentToolingService.disposeApplication(applicationId);
    }
  }

  @Test
  public void connectivityTestingForConfigDeclaredOnDomainUsingDomainId() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");
    String domainId = restAgentToolingService.deployDomain(emailDomainLocation, getDeploymentProperties());

    assertThat(domainId, is(notNullValue()));

    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain-exploded");
    String applicationId = restAgentToolingService.deployApplication(emailAppLocation, domainId, getDeploymentProperties());

    assertThat(applicationId, is(notNullValue()));

    try {
      final ConnectionValidationResult connectionValidationResult =
          restAgentToolingService.testConnectionDomain(domainId, "httpListenerConfig", -1);
      assertThat(connectionValidationResult.isValid(), is(true));
    } finally {
      restAgentToolingService.disposeApplication(applicationId);
      restAgentToolingService.disposeDomain(domainId);
    }
  }

  @Test
  @Description("Checks connectivity testing for an element declared on an application, checking that reconnection strategy is not honoured")
  public void connectivityTestingConfigDeclaredOnAppWithReconnectionNotHonoured() throws URISyntaxException {
    checkNoReconnectionOnTooling("applications/reconnect-forever-fail-deployment");
    checkNoReconnectionOnTooling("applications/reconnect-forever-no-fail-deployment");
    checkNoReconnectionOnTooling("applications/reconnect-few-fail-deployment");
    checkNoReconnectionOnTooling("applications/reconnect-few-no-fail-deployment");
  }


  private void checkNoReconnectionOnTooling(String appLocation) throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File location = new File(targetTestClassesFolder, appLocation);

    String applicationID = restAgentToolingService.deployApplication(location, getDeploymentProperties());
    setUpEmailServer();
    try {
      assertThat(applicationID, is(notNullValue()));
      final ConnectionValidationResult result =
          restAgentToolingService.testConnectionApplication(applicationID, "emailConfig", -1);
      assertThat(result.isValid(), is(false));
      assertThat(result.getMessage(), containsString("Couldn't connect to host"));
    } finally {
      try {
        restAgentToolingService.disposeApplication(applicationID);
      } catch (Exception e) {
      }
      stopEmailServer();
    }
  }

}
