/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client.tooling.domains;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_OCTET_STREAM_TYPE;
import static javax.ws.rs.core.Response.Status.Family.SUCCESSFUL;
import static org.glassfish.jersey.client.ClientProperties.READ_TIMEOUT;
import static org.mule.tooling.agent.rest.client.exceptions.ToolingServiceAPIException.newToolingServiceAPIException;
import static org.mule.tooling.agent.rest.client.tooling.Tooling.VERBOSE_QUERY_PARAM;

import org.mule.tooling.agent.rest.client.exceptions.ToolingServiceAPIException;
import org.mule.tooling.agent.rest.client.tooling.applications.model.ApplicationsGETResponse;
import org.mule.tooling.agent.rest.client.tooling.domains.domainId.DomainId;
import org.mule.tooling.agent.rest.client.tooling.domains.model.DomainsGETResponse;
import org.mule.tooling.agent.rest.client.tooling.domains.model.DomainsPUTBody;
import org.mule.tooling.agent.rest.client.tooling.domains.model.DomainsPUTResponse;

import java.io.InputStream;
import java.util.List;

import javax.ws.rs.client.Client;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.GenericType;
import javax.ws.rs.core.Response;

/**
 * Application web resource.
 *
 * @since 4.0
 */
public class Domains {

  private static final String ARTIFACT_ID = "artifactId";

  private String baseUrl;
  private Client client;

  public Domains(String baseUrl, Client client) {
    this.baseUrl = (baseUrl + "/domains");
    this.client = client;
  }

  private Client getClient() {
    return this.client;
  }

  private String getBaseUri() {
    return baseUrl;
  }

  /**
   * GET operation to retrieve the list of domains deployed.
   *
   * @param verboseErrorsEnabled  flag that defines if it has to be verbose the response in case of errors/failures.
   * @param readTimeout           read timeout in milliseconds
   * @return {@link List} of {@link ApplicationsGETResponse}s
   * @throws ToolingServiceAPIException if there was an error.
   */
  public List<DomainsGETResponse> get(boolean verboseErrorsEnabled, int readTimeout) throws ToolingServiceAPIException {
    WebTarget target = this.getClient().target(getBaseUri());
    target = target.queryParam(VERBOSE_QUERY_PARAM, verboseErrorsEnabled);
    final javax.ws.rs.client.Invocation.Builder invocationBuilder = target.request(APPLICATION_JSON_TYPE);
    if (readTimeout > 0) {
      invocationBuilder.property(READ_TIMEOUT, readTimeout);
    }
    Response response = invocationBuilder.get();
    if (response.getStatusInfo().getFamily() != SUCCESSFUL) {
      throw newToolingServiceAPIException(response);
    }
    return response.readEntity(new GenericType<List<DomainsGETResponse>>() {

    });
  }

  /**
   * PUT operation to deploy a tooling application not compressed.
   *
   * @param body                  {@link DomainsPUTBody} request body.
   * @param verboseErrorsEnabled  flag that defines if it has to be verbose the response in case of errors/failures.
   * @param readTimeout           read timeout in milliseconds
   * @return {@link DomainsPUTResponse} for the deployed application.
   * @throws ToolingServiceAPIException if there was an error.
   */
  public DomainsPUTResponse put(DomainsPUTBody body, boolean verboseErrorsEnabled, int readTimeout)
      throws ToolingServiceAPIException {
    WebTarget target = this.getClient().target(getBaseUri());
    target = target.queryParam(VERBOSE_QUERY_PARAM, verboseErrorsEnabled);
    final javax.ws.rs.client.Invocation.Builder invocationBuilder = target.request(APPLICATION_JSON_TYPE);
    if (readTimeout > 0) {
      invocationBuilder.property(READ_TIMEOUT, readTimeout);
    }
    Response response = invocationBuilder.put(javax.ws.rs.client.Entity.json(body));
    if (response.getStatusInfo().getFamily() != SUCCESSFUL) {
      throw newToolingServiceAPIException(response);
    }
    return response.readEntity(DomainsPUTResponse.class);
  }

  /**
   * PUT operation to deploy a tooling application compressed from an {@link InputStream}.
   *
   * @param body {@link InputStream} to read the content of the application zipped.
   * @param verboseErrorsEnabled  flag that defines if it has to be verbose the response in case of errors/failures.
   * @param readTimeout           read timeout in milliseconds
   * @return {@link DomainsPUTResponse} for the deployed application.
   * @throws ToolingServiceAPIException if there was an error.
   */
  public DomainsPUTResponse put(InputStream body, String artifactId, boolean verboseErrorsEnabled, int readTimeout)
      throws ToolingServiceAPIException {
    WebTarget target = this.getClient().target(getBaseUri());
    target = target.queryParam(VERBOSE_QUERY_PARAM, verboseErrorsEnabled);
    target = target.queryParam(ARTIFACT_ID, artifactId);
    final javax.ws.rs.client.Invocation.Builder invocationBuilder = target.request(APPLICATION_JSON_TYPE);
    if (readTimeout > 0) {
      invocationBuilder.property(READ_TIMEOUT, readTimeout);
    }
    Response response = invocationBuilder.put(javax.ws.rs.client.Entity.entity(body, APPLICATION_OCTET_STREAM_TYPE));
    if (response.getStatusInfo().getFamily() != SUCCESSFUL) {
      throw newToolingServiceAPIException(response);
    }
    return response.readEntity(DomainsPUTResponse.class);
  }

  /**
   * Access to the domain resource.
   *
   * @param domainId the identifier of the domain.
   * @return {@link DomainId} to operate with.
   */
  public final DomainId domainId(String domainId) {
    return new DomainId(getBaseUri(), getClient(), domainId);
  }

}
