/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.agent.rest.client;


import static java.util.Collections.emptyMap;
import static org.hamcrest.CoreMatchers.is;
import static org.junit.Assert.assertThat;
import static org.mule.tooling.client.test.utils.matchers.DataWeaveMatcher.WeaveMatcherField.newBuilder;
import static org.mule.tooling.client.test.utils.matchers.DataWeaveMatcher.weaveMatcher;
import org.mule.tooling.client.test.RuntimeType;

import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewRequest;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewResponse;

import java.io.File;
import java.net.URISyntaxException;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;

@Feature("DataWeave")
@Story("Runner support with application context")
public class DataWeaveRunnerTestCase extends AbstractRestAgentToolingServiceTestCase {

  public DataWeaveRunnerTestCase(RuntimeType runtimeType) {
    super(runtimeType);
  }

  @Test
  @Description("Checks if dataweave can access to an application class")
  public void runDataWeaveWithinAppContext() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File pojoApplication = new File(targetTestClassesFolder, "applications/pojo");

    String applicationId = restAgentToolingService.deployApplication(null, pojoApplication, emptyMap());

    PreviewRequest request = new PreviewRequest();
    request.setScript("output application/java\n"
        + "---\n"
        + "{\n"
        + "name: 'Matias',\n"
        + "age: 29\n"
        + "} as Object { class: \"pojo.Poyito\" }");

    PreviewResponse response = restAgentToolingService.runDataWeaveApplication(applicationId, request);

    assertThat(response.getResult(), weaveMatcher("pojo.Poyito",
                                                  newBuilder()
                                                      .withFieldName("age")
                                                      .withFieldValue("29")
                                                      .asType("Number")
                                                      .withClassName(Integer.class.getName())
                                                      .build(),
                                                  newBuilder()
                                                      .withFieldName("name")
                                                      .withFieldValue("\"Matias\"")
                                                      .asType("String")
                                                      .withClassName(String.class.getName())
                                                      .build()));
  }

  @Test
  @Description("Checks dataweave resolution in agent")
  public void runDataWeaveSimpleScript() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File pojoApplication = new File(targetTestClassesFolder, "applications/pojo");

    String applicationId = restAgentToolingService.deployApplication(null, pojoApplication, emptyMap());

    PreviewRequest request = new PreviewRequest();
    request.setScript("output application/json\n"
        + "---\n"
        + "{\n"
        + "name: 'Matias',\n"
        + "age: 29\n"
        + "}");

    String expected = "{\n"
        + "  \"name\": \"Matias\",\n"
        + "  \"age\": 29\n"
        + "}";

    PreviewResponse response = restAgentToolingService.runDataWeaveApplication(applicationId, request);
    assertThat(response.getResult(), is(expected));
  }

  @Test
  @Description("Checks dataweave resolution in agent even when domain artifacts are used")
  public void runDataWeaveSimpleScriptOnDomainArtifact() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File pojoDomain = new File(targetTestClassesFolder, "domains/pojo-domain");

    String domainId = restAgentToolingService.deployDomain(null, pojoDomain, emptyMap());

    PreviewRequest request = new PreviewRequest();
    request.setScript("output application/json\n"
        + "---\n"
        + "{\n"
        + "name: 'Matias',\n"
        + "age: 29\n"
        + "}");

    String expected = "{\n"
        + "  \"name\": \"Matias\",\n"
        + "  \"age\": 29\n"
        + "}";

    PreviewResponse response = restAgentToolingService.runDataWeaveDomain(domainId, request);
    assertThat(response.getResult(), is(expected));
  }

}
