/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent;

import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.api.value.ValueResult;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationName.messageHistory.AgentTrackingNotificationResponse;
import org.mule.tooling.client.api.configuration.agent.proxy.ProxyConfig;
import org.mule.tooling.client.api.configuration.ssl.SslConfiguration;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingObjectNotFoundException;
import org.mule.tooling.client.api.connectivity.UnsupportedConnectivityTestingObjectException;
import org.mule.tooling.client.api.exception.DeploymentException;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;

import com.mulesoft.agent.domain.tooling.BundleDescriptor;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewRequest;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewResponse;
import com.mulesoft.agent.domain.tracking.AgentTrackingNotification;

import java.io.File;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Provides tooling operations for applications. An application may be a Mule Application or a Domain.
 *
 * @since 4.0
 */
public interface RuntimeToolingService {

  /**
   * Sets {@link URL} for the Remote Mule Runtime Agent Tooling API to configure the client.
   *
   * @param url                   {@link URL} to the base URL for the Mule Runtime Agent.
   * @param defaultConnectTimeout default connection timeout for REST client.
   * @param defaultReadTimeout    default read timeout for REST client.
   * @param sslConfiguration      {@link Optional} {@link SslConfiguration} for the Mule Runtime Agent.
   * @param proxyConfig           {@link Optional} {@link ProxyConfig} for the Mule Runtime Agent.
   * @param authorizationToken    Token to communicate with agent securely.
   * @throws IllegalStateException if a baseUrl was already set.
   */
  void setToolingApiUrl(URL url,
                        long defaultConnectTimeout,
                        long defaultReadTimeout,
                        Optional<SslConfiguration> sslConfiguration,
                        Optional<ProxyConfig> proxyConfig,
                        String authorizationToken);

  /**
   * @param muleVersion defines the API version to be called.
   */
  void setMuleVersion(String muleVersion);

  /**
   * @return {@code true} if the remote tooling service is operational and ready to be used.
   */
  boolean isOperational();

  /**
   * Deploys an application to the Mule Runtime from the given {@code appLocation} that points to the exploded application
   * content.
   * <p>
   * The application content is copied from the specified location, be aware that the original files are not modified
   * or watched later.
   *
   * @param id predefined identifier for the application deployment, if Runtime Agent doesn't support it an autogenerated
   *           identifier will be returned, if not the same.
   * @param appLocation {@link File} to the application exploded content.
   * @param domainName {@link String} to the domain name to be linked to when deploying this application. Can be null.
   * @return {@link String} application identifier for the application deployed.
   * @throws DeploymentException         if there was an error while deploying the application.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  String deployApplication(String id, File appLocation, String domainName, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException;

  /**
   * Deploys an application to the Mule Runtime from the given {@code appLocation} that points to the exploded application
   * content.
   * <p>
   * The application content is copied from the specified location, be aware that the original files are not modified
   * or watched later.
   *
   * @param id predefined identifier for the application deployment, if Runtime Agent doesn't support it an autogenerated
   *           identifier will be returned, if not the same. Can be null.
   * @param appLocation {@link File} to the application exploded content.
   * @return {@link String} application identifier for the application deployed.
   * @throws DeploymentException         if there was an error while deploying the application.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  default String deployApplication(String id, File appLocation, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException {
    return deployApplication(id, appLocation, null, deploymentProperties);
  }

  /**
   * Deploys a domain to the Mule Runtime from the given {@code domainLocation} that points to the exploded domain
   * content.
   * <p>
   * The domain content is copied from the specified location, be aware that the original files are not modified
   * or watched later.
   *
   * @param id predefined identifier for the application deployment, if Runtime Agent doesn't support it an autogenerated
   *           identifier will be returned, if not the same. Can be null.
   * @param domainLocation {@link File} to the domain exploded content.
   * @return {@link String} application identifier for the domain deployed.
   * @throws DeploymentException         if there was an error while deploying the domain.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  String deployDomain(String id, File domainLocation, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException;

  /**
   * Deploys an application to the Mule Runtime with the content of the application jar from the {@code inputStream}.
   *
   * @param id predefined identifier for the application deployment, if Runtime Agent doesn't support it an autogenerated
   *           identifier will be returned, if not the same. Can be null.
   * @param inputStream {@link InputStream} with the content of the application jar file.
   * @return {@link String} application identifier for the application deployed.
   * @throws DeploymentException         if there was an error while deploying the application.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  String deployApplication(String id, InputStream inputStream, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException;

  /**
   * Deploys an application to the Mule Runtime with the content of the application jar from the {@code inputStream}.
   *
   * @param id predefined identifier for the application deployment, if Runtime Agent doesn't support it an autogenerated
   *           identifier will be returned, if not the same. Can be null.
   * @param inputStream {@link InputStream} with the content of the application jar file.
   * @param domainName {@link String} to the domain name to be linked to when deploying this application. Can be null.
   * @return {@link String} application identifier for the application deployed.
   * @throws DeploymentException         if there was an error while deploying the application.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  String deployApplication(String id, InputStream inputStream, String domainName, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException;

  /**
   * Deploys a domain to the Mule Runtime with the content of the application jar from the {@code inputStream}.
   *
   * @param id predefined identifier for the application deployment, if Runtime Agent doesn't support it an autogenerated
   *           identifier will be returned, if not the same. Can be null.
   * @param inputStream {@link InputStream} with the content of the domain jar file.
   * @return {@link String} application identifier for the domain deployed.
   * @throws DeploymentException         if there was an error while deploying the domain.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  String deployDomain(String id, InputStream inputStream, Map<String, String> deploymentProperties)
      throws DeploymentException, ServiceUnavailableException;

  /**
   * Disposes the application for the given {@code applicationId}.
   *
   * @param applicationId {@link String} identifies the application to be disposed.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   * @throws ToolingException            if an unexpected error happens.
   */
  void disposeApplication(String applicationId) throws ServiceUnavailableException;

  /**
   * Disposes the domain for the given {@code domainId}.
   *
   * @param domainId {@link String} identifies the domain to be disposed.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   * @throws ToolingException            if an unexpected error happens.
   */
  void disposeDomain(String domainId) throws ServiceUnavailableException;

  /**
   * Does connection testing over the {@code componentId} given for the application with the {@code applicationId} identifier.
   *
   * @param applicationId {@link String} identifies the application to be tested.
   * @param componentId   {@link String} identifies the component to be tested.
   * @param readTimeout   read timeout in milliseconds
   * @return {@link ConnectionValidationResult} with the result of the test.
   * @throws NoSuchApplicationException                    if the application is not deployed or it was evicted by the the service (free resources).
   * @throws UnsupportedConnectivityTestingObjectException when it's not possible to do connectivity testing over the mule
   *                                                       component.
   * @throws ConnectivityTestingObjectNotFoundException    when the object to use to do connectivity does not exists.
   * @throws ToolingException                              if an unexpected error happens.
   * @throws ServiceUnavailableException                   if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  ConnectionValidationResult testConnectionApplication(String applicationId, String componentId, long readTimeout)
      throws NoSuchApplicationException, UnsupportedConnectivityTestingObjectException,
      ConnectivityTestingObjectNotFoundException, ServiceUnavailableException;

  /**
   * Does connection testing over the {@code componentId} given for the domain with the {@code domainId} identifier.
   *
   * @param domainId {@link String} identifies the domain to be tested.
   * @param componentId   {@link String} identifies the component to be tested.
   * @param readTimeout   read timeout in milliseconds
   * @return {@link ConnectionValidationResult} with the result of the test.
   * @throws NoSuchApplicationException                    if the domain is not deployed or it was evicted by the the service (free resources).
   * @throws UnsupportedConnectivityTestingObjectException when it's not possible to do connectivity testing over the mule
   *                                                       component.
   * @throws ConnectivityTestingObjectNotFoundException    when the object to use to do connectivity does not exists.
   * @throws ToolingException                              if an unexpected error happens.
   * @throws ServiceUnavailableException                   if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  ConnectionValidationResult testConnectionDomain(String domainId, String componentId, long readTimeout)
      throws NoSuchApplicationException, UnsupportedConnectivityTestingObjectException,
      ConnectivityTestingObjectNotFoundException, ServiceUnavailableException;

  /**
   * Does connection testing from a temporary application with the given configuration and extensions for the componentId.
   *
   * @param dependencies          {@link BundleDescriptor}'s libraries to be used by the application.
   * @param artifactDeclaration   {@link ArtifactDeclaration} defines the application configuration.
   * @param componentId           {@link String} identifies the component to be tested.
   * @param readTimeout           read timeout in milliseconds
   * @return {@link ConnectionValidationResult} with the result of the test.
   * @throws UnsupportedConnectivityTestingObjectException when it's not possible to do connectivity testing over the mule
   *                                                       component.
   * @throws ConnectivityTestingObjectNotFoundException}   when the object to use to do connectivity does not exists.
   * @throws ToolingException                              if an unexpected error happens.
   * @throws ServiceUnavailableException                   if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  ConnectionValidationResult testConnection(List<BundleDescriptor> dependencies, ArtifactDeclaration artifactDeclaration,
                                            String componentId, long readTimeout)
      throws UnsupportedConnectivityTestingObjectException, ConnectivityTestingObjectNotFoundException,
      ServiceUnavailableException;


  /**
   * Returns a {@link MetadataResult} of {@link org.mule.runtime.api.metadata.MetadataKey} described by {@link
   * org.mule.runtime.api.metadata.MetadataKeyProvider#getMetadataKeys()} of the associated {@link
   * org.mule.runtime.api.metadata.MetadataKeyProvider} component identified by the given applicationId and componentLocation.
   *
   * @param applicationId {@link String} identifies the application.
   * @param componentLocation the location path of the {@link org.mule.runtime.api.metadata.MetadataKeyProvider} component inside the flow.
   * @param readTimeout           read timeout in milliseconds
   * @return Successful {@link MetadataResult} if the keys are successfully resolved Failure.
   * {@link MetadataResult} if there is an error while resolving the keys.
   * @throws ToolingException            if an unexpected error happens.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  MetadataResult<MetadataKeysContainer> getMetadataKeysApplication(String applicationId, String componentLocation,
                                                                   long readTimeout)
      throws ServiceUnavailableException;

  /**
   * Returns a {@link MetadataResult} of {@link org.mule.runtime.api.metadata.MetadataKey} described by {@link
   * org.mule.runtime.api.metadata.MetadataKeyProvider#getMetadataKeys()} of the associated {@link
   * org.mule.runtime.api.metadata.MetadataKeyProvider} component identified by the given domainId and componentLocation.
   *
   * @param domainId {@link String} identifies the domain.
   * @param componentLocation the location path of the {@link org.mule.runtime.api.metadata.MetadataKeyProvider} component inside the flow.
   * @param readTimeout           read timeout in milliseconds
   * @return Successful {@link MetadataResult} if the keys are successfully resolved Failure.
   * {@link MetadataResult} if there is an error while resolving the keys.
   * @throws ToolingException            if an unexpected error happens.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  MetadataResult<MetadataKeysContainer> getMetadataKeysDomain(String domainId, String componentLocation, long readTimeout)
      throws ServiceUnavailableException;

  /**
   * Resolves the Metadata description for the operation message processor component identified by the given applicationId and
   * componentLocaton.
   *
   * @param applicationId {@link String} identifies the application.
   * @param componentLocation the location path of the operation message processor component inside the
   *        flow.
   * @param readTimeout           read timeout in milliseconds
   * @return An {@link ComponentMetadataDescriptor} with the Metadata representation of the Component. Successful
   *         {@link MetadataResult} if the Metadata is successfully retrieved. Failure {@link MetadataResult} when the Metadata
   *         retrieval of any element fails for any reason
   * @throws ToolingException if an unexpected error happens.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(String applicationId, String componentLocation,
                                                                                   long readTimeout)
      throws ServiceUnavailableException;

  /**
   * Resolves the Metadata description for the source component of the given applicationId and componentLocation.
   *
   * @param applicationId {@link String} identifies the application.
   * @param componentLocation the location of the component to be queried as {@link org.mule.runtime.api.metadata.MetadataKeyProvider} component.
   * @param readTimeout           read timeout in milliseconds
   * @return An {@link ComponentMetadataDescriptor} with the Metadata representation of the Component. Successful
   *         {@link MetadataResult} if the Metadata is successfully retrieved. Failure {@link MetadataResult} when the Metadata
   *         retrieval of any element fails for any reason
   * @throws ToolingException if an unexpected error happens.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   */
  MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(String applicationId, String componentLocation,
                                                                             long readTimeout)
      throws ServiceUnavailableException;

  /**
   * Disposes the metadata associated with the hashKey in the Metadata cache on Mule Runtime side.
   *
   * @param applicationId {@link String} identifies the application.
   * @param hashKey could be generated for a component or just for a particular component.
   */
  void disposeApplicationMetadataCache(String applicationId, String hashKey);

  /**
   * Disposes the metadata associated with the hashKey in the Metadata cache on Mule Runtime side.
   *
   * @param domainId {@link String} identifies the domain.
   * @param hashKey could be generated for a component or just for a particular component.
   */
  void disposeDomainMetadataCache(String domainId, String hashKey);

  /**
   * Enables an application for message history.
   *
   * @param applicationName the name of the application to be used when deploying it.
   * @throws NoSuchApplicationException if the application to be enable for message history is not already deployed.
   */
  void enableMessageHistory(String applicationName) throws NoSuchApplicationException;

  /**
   * Disables message history for the given application.
   *
   * @param applicationName the name of the application to be disabled.
   */
  void disableMessageHistory(String applicationName);

  /**
   * Consumes the message history notifications buffered by Mule Runtime for the given application.
   *
   * @param applicationName the name of the application to get its message history notifications.
   * @param chunkSize the maximum number of notifications to be consumed by this request.
   * @return {@link List} of {@link AgentTrackingNotification} collected by the remote service at this moment.
   */
  List<AgentTrackingNotificationResponse> consumeMessageHistoryNotifications(String applicationName, int chunkSize);

  /**
   * Execute a dataWeave script in the context of an application.
   *
   * @param applicationId application that provides context
   * @param request parameters for running preview
   * @return a wrapper that describes execution result or error details if execution failed.
   */
  PreviewResponse runDataWeaveApplication(String applicationId, PreviewRequest request);

  /**
   * Execute a dataWeave script in the context of an application.
   *
   * @param applicationId application that provides context
   * @param request parameters for running preview
   * @return a wrapper that describes execution result or error details if execution failed.
   */
  PreviewResponse runDataWeaveDomain(String applicationId, PreviewRequest request);

  /**
   * Resolves the {@link Value values} for a component's value provider located in the given {@link Location}.
   *
   * @param location     The {@link Location} where the configuration can be found.
   * @param providerName The name of the value provider to resolve the {@link Value values}.
   * @return the {@link ValueResult result} of the resolving of the values.
   * @see ValueResult
   */
  ValueResult getValuesApplication(String applicationId, String location, String providerName);

  /**
   * Resolves the {@link Value values} for a component's value provider located in the given {@link Location}.
   *
   * @param location     The {@link Location} where the configuration can be found.
   * @param providerName The name of the value provider to resolve the {@link Value values}.
   * @return the {@link ValueResult result} of the resolving of the values.
   * @see ValueResult
   */
  ValueResult getValuesDomain(String domainId, String location, String providerName);

}
