/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client.api;

import static com.fasterxml.jackson.databind.DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES;
import static javax.ws.rs.core.HttpHeaders.AUTHORIZATION;
import static jersey.repackaged.com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.glassfish.jersey.client.ClientProperties.CONNECT_TIMEOUT;
import static org.glassfish.jersey.client.ClientProperties.PROXY_URI;
import static org.glassfish.jersey.client.ClientProperties.READ_TIMEOUT;

import org.mule.tooling.agent.rest.client.filter.AuthorizationRequestFilter;
import org.mule.tooling.agent.rest.client.tooling.Tooling;
import org.mule.tooling.client.api.configuration.agent.proxy.ProxyConfig;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URLConnection;
import java.util.Optional;

import javax.net.ssl.SSLContext;
import javax.ws.rs.client.Client;
import javax.ws.rs.client.ClientBuilder;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;

import com.fasterxml.jackson.jaxrs.json.JacksonJaxbJsonProvider;

import org.glassfish.jersey.client.ClientConfig;
import org.glassfish.jersey.client.HttpUrlConnectorProvider;
import org.glassfish.jersey.filter.LoggingFilter;

/**
 * REST client implementation for Tooling REST API from Mule Runtime Agent.
 *
 * @since 4.0
 */
public class ToolingServiceAPIClient {

  public final Tooling tooling;

  private ToolingServiceAPIClient(String toolingApiUrl, int defaultConnectTimeout, int defaultReadTimeout,
                                  Optional<SSLContext> sslContext, Optional<ProxyConfig> proxyConfig, String authorizationToken) {
    checkNotNull(toolingApiUrl, "baseUrl cannot be null");

    tooling = new Tooling(toolingApiUrl,
                          newClient(defaultConnectTimeout, defaultReadTimeout, sslContext, proxyConfig, authorizationToken));
  }

  private Client newClient(int connectTimeout,
                           int readTimeout,
                           Optional<SSLContext> sslContextOptional,
                           Optional<ProxyConfig> proxyConfig,
                           String authorizationToken) {

    ClientConfig configuration = new ClientConfig(new JacksonJaxbJsonProvider().configure(FAIL_ON_UNKNOWN_PROPERTIES, false));
    configuration.property(CONNECT_TIMEOUT, connectTimeout);
    configuration.property(READ_TIMEOUT, readTimeout);

    if (proxyConfig.isPresent()) {
      ProxyConfig proxyConfiguration = proxyConfig.get();
      String proxyUri = "http://" + proxyConfiguration.getHost();
      if (proxyConfiguration.getPort() != null) {
        proxyUri = proxyUri + ":" + proxyConfiguration.getPort();
      }
      configuration.property(PROXY_URI, proxyUri);
    }

    HttpUrlConnectorProvider connectorProvider = new HttpUrlConnectorProvider();
    connectorProvider.connectionFactory(url -> {
      Proxy proxy = Proxy.NO_PROXY;
      if (proxyConfig.isPresent()) {
        proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyConfig.get().getHost(), proxyConfig.get().getPort()));
      }
      URLConnection urlConnection = url.openConnection(proxy);
      return (HttpURLConnection) urlConnection;
    });

    configuration.connectorProvider(connectorProvider);

    ClientBuilder builder = ClientBuilder.newBuilder();

    configuration.register(new LoggingFilter());
    configuration.register(new AuthorizationRequestFilter(authorizationToken));
    builder.withConfig(configuration);

    sslContextOptional.ifPresent(sslContext -> builder.sslContext(sslContext));

    return builder.build();
  }

  /**
   * Factory method that creates the REST client.
   *
   * @param baseUrl               base URL for the service. Must not be {@code null}.
   * @param defaultConnectTimeout default connection timeout for REST client.
   * @param defaultReadTimeout    default read timeout for REST client.
   * @param sslContext            ssl configuration for the REST client.
   * @param proxyConfig           proxyConfig for the REST client.
   * @param authorizationToken    token to safely communicate with the Agent
   * @return {@link ToolingServiceAPIClient} to access REST operations.
   */
  public static ToolingServiceAPIClient create(String baseUrl,
                                               int defaultConnectTimeout, int defaultReadTimeout,
                                               Optional<SSLContext> sslContext,
                                               Optional<ProxyConfig> proxyConfig,
                                               String authorizationToken) {
    return new ToolingServiceAPIClient(baseUrl, defaultConnectTimeout, defaultReadTimeout, sslContext, proxyConfig,
                                       authorizationToken);
  }

}
