/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static org.apache.commons.lang3.exception.ExceptionUtils.getStackTrace;

import org.mule.runtime.api.message.ErrorType;
import org.mule.tooling.client.api.connectivity.ConnectionValidationResult;

/**
 * Immutable implementation of {@link ConnectionValidationResult}.
 *
 * @since 4.0
 */
public final class ConnectionValidationResultFactory {

  /**
   * Creates an instance of the connectivity testing result.
   *
   * @param validationStatus {@code true} if the connectivity testing was successful.
   * @param message          {@link String} describing the reason why the connectivity testing failure.
   * @param errorType        {@link ErrorType} the error type representing the connectivity testing failure.
   * @param exception        {@link String} with the exception stack trace that cause the conectivity testing to fail.
   */
  private static ConnectionValidationResult newInstance(boolean validationStatus, String message, ErrorType errorType,
                                                        String exception) {
    ConnectionValidationResult result = new ConnectionValidationResult();
    result.setValid(validationStatus);
    result.setMessage(message);
    result.setErrorType(toErrorTypeDTO(errorType));
    result.setException(exception);
    return result;
  }

  /**
   * @return {@link ConnectionValidationResult} successful.
   */
  public static ConnectionValidationResult success() {
    return newInstance(true, null, null, null);
  }

  /**
   * @return {@link ConnectionValidationResult} failure with the error message and code.
   */
  public static ConnectionValidationResult failure(String message, ErrorType errorType, String exception) {
    return newInstance(false, message, errorType, exception);
  }

  /**
   * @return {@link ConnectionValidationResult} failure with the error message and code.
   */
  public static ConnectionValidationResult failure(String message, ErrorType errorType, Throwable cause) {
    return newInstance(false, message, errorType, getStackTrace(cause));
  }

  private static org.mule.tooling.client.api.error.ErrorType toErrorTypeDTO(ErrorType errorType) {
    if (errorType == null) {
      return null;
    }
    return new org.mule.tooling.client.api.error.ErrorType(errorType.getIdentifier(), errorType.getNamespace(),
                                                           toErrorTypeDTO(errorType.getParentErrorType()));
  }

}
