/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.agent.rest.client;

import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;

import io.qameta.allure.Description;
import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;

public abstract class DeployApplicationTestCase extends AbstractDeploymentTestCase {

  @Rule
  public ExpectedException expectedException = none();
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Before
  public void beforeTest() {
    setUpEmailServer();
  }

  @After
  public void afterTest() {
    stopEmailServer();
  }

  @Test
  @Description("Checks Deployment of a Tooling Application from Mule Runtime Agent using the RuntimeToolingService")
  public void doDeployApplication() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    String applicationId = doDeployApplication(emailAppLocation);

    assertThat(applicationId, is(notNullValue()));
  }

  @Test
  public void deployApplicationWithDomainExploded() throws URISyntaxException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");
    String domainId = doDeployDomain(emailDomainLocation);

    assertThat(domainId, is(notNullValue()));

    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain-exploded");
    String applicationId = doDeployApplication(emailAppLocation, domainId);

    assertThat(applicationId, is(notNullValue()));

    restAgentToolingService.disposeApplication(applicationId);
    restAgentToolingService.disposeDomain(domainId);
  }

  @Test
  @Description("Checks Deployment of a Tooling Application using InputStream from Mule Runtime Agent using the RuntimeToolingService")
  public void deployApplicationInputStream() throws Exception {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailAppLocation = new File(targetTestClassesFolder, "applications/email");

    File targetFile = temporaryFolder.newFile();
    compress(targetFile, emailAppLocation);

    String applicationId = doDeployApplication(new FileInputStream(targetFile));

    assertThat(applicationId, is(notNullValue()));
  }

  @Test
  public void deployApplicationWithDomainExplodedUsingInputStream() throws URISyntaxException, IOException {
    File targetTestClassesFolder = new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().toURI());
    File emailDomainLocation = new File(targetTestClassesFolder, "domains/email-domain-exploded");

    File domainTargetFile = temporaryFolder.newFile();
    compress(domainTargetFile, emailDomainLocation);

    String domainId = doDeployDomain(new FileInputStream(domainTargetFile));

    assertThat(domainId, is(notNullValue()));

    File emailAppLocation = new File(targetTestClassesFolder, "applications/email-with-domain-exploded");

    File appTargetFile = temporaryFolder.newFile();
    compress(appTargetFile, emailAppLocation);


    String applicationId = doDeployApplication(new FileInputStream(appTargetFile), domainId);

    assertThat(applicationId, is(notNullValue()));

    restAgentToolingService.disposeApplication(applicationId);
    restAgentToolingService.disposeDomain(domainId);
  }

  protected abstract String doDeployDomain(File location);

  protected abstract String doDeployDomain(InputStream inputStream);

  protected abstract String doDeployApplication(File emailAppLocation);

  protected abstract String doDeployApplication(File location, String domainId);

  protected abstract String doDeployApplication(InputStream inputStream);

  protected abstract String doDeployApplication(InputStream inputStream, String domainId);

}
